/* v1.1
 *
 * sensors.c:  Sensor code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "class.h"
#include "output.h"
#include "object.h"
#include "sensors.h"
#include "damage.h"
#include "smisc.h"
#include "shields.h"
#include "events.h"
#include "tactical.h"
#include "nav.h"

#define Shield_string(ship, shield) (ship->shield_status[shield] == SHLD_UP ? "UP  " : "DOWN")

/*
 * Macros for accessing the next and previous entries in the
 * range sorted linked list
 */
#define RNEXT(tag) (tag->range_next)
#define RPREV(tag) (tag->range_prev)

enum contact_value {NOT_SENSED, SENSED};

static int check_pair(TAG *, TAG *);
static void sense_huge_object(TAG *, TAG *);
static TAG *nearest_huge_object(TAG *);
static void new_sensor_turn(int);
static void cleanup_contacts(int);
static void update_contact(TAG *, TAG *, int);
static CONTACT *add_contact(TAG *source, TAG *contact);
static void new_contact(TAG *, CONTACT *);
static void contact_lost(TAG *, CONTACT *, int);

static void snsFormatShortSensorInfo(TAG *senser, CONTACT *contact, SSTR *ps,
				     struct timeval *now);
static void build_contact_name(char *name, CONTACT *contact, int buflen);

static TAG* snsFindInsertPosPrev(TAG *tag, range_t new_range);
static TAG* snsFindInsertPosNext(TAG *tag, range_t new_range);

/* Used for detecting stale contacts */
static int sensor_turn_id;

/* Sensor routines */

/*
 * snsBuildDistanceTables:  Build and update a double-linked list of tags
 *                          ordered by the object range from the hub (0,0,0).
 */
void snsBuildDistanceTables(int space)
{
    XYZ     pos;
    TAG    *tag, *itag;
    range_t new_range;
    struct timeval now;
    
#ifdef ENABLE_DEBUGGING
    /* Debugging feature - should pick up any weird features immediately.
       The range parameter should not be modified anywhere except here,
       so this one should work. The only possibility should be a newly
       added object failing the very first time. */
    snsSanityCheck(space);
#endif
    
    gettimeofday(&now, NULL);

    /*
     * Cycle through the space object list (not changed during this loop)
     * and incrementally update the range sorted linked list of object.
     */
    for (tag=space_info[space].list; tag != NULL; tag = tag->next) {
	
	/*
	 * Don't process removed objects, or objects that have not
	 * moved this update cycle
	 */
	if (Removed(tag) || (tag->moveID != move_turn_id))
	    continue;
	
	object_position(tag, &pos, &now, 0);

	new_range = HubRange(pos);
	
	if ((new_range < tag->range) && (RPREV(tag) != NULL)) {
	    
	    /* Search towards the head of the list */
	    itag = snsFindInsertPosPrev(tag, new_range);
	    
	    if (itag != RPREV(tag)) {

		/* Update the listed position */
		RNEXT(RPREV(tag)) = RNEXT(tag);
		
		if (RNEXT(tag) != NULL)
		    RPREV(RNEXT(tag)) = RPREV(tag);

		if (itag == NULL) {
		    RNEXT(tag) = space_info[space].dist;
		    space_info[space].dist = tag;
		}
		else {
		    RNEXT(tag) = RNEXT(itag);
		    RNEXT(itag) = tag;
		}
		
		RPREV(tag) = itag;
		
		if (RNEXT(tag) != NULL)
		    RPREV(RNEXT(tag)) = tag;
	    }
	    
	} else if ((new_range > tag->range) && (RNEXT(tag) != NULL)) {
	    
	    /* Search towards the tail of the list */
	    itag = snsFindInsertPosNext(tag, new_range);
	    
	    if (itag != tag) {
		
		/* Update the listed position */
		if (RPREV(tag) == NULL)
		    space_info[space].dist = RNEXT(tag);
		else
		    RNEXT(RPREV(tag)) = RNEXT(tag);
		
		RPREV(RNEXT(tag)) = RPREV(tag);
		
		RNEXT(tag) = RNEXT(itag);
		RPREV(tag) = itag;
		RNEXT(itag) = tag;
		if (RNEXT(tag) != NULL)
		    RPREV(RNEXT(tag)) = tag;
	    }
	}
	
	/* Update the range field and the movement ID to show the
	 * object is uptodate */
	tag->moveID --;
	tag->range = new_range;
    }
    
#ifdef ENABLE_DEBUGGING
    /* Debugging feature - should pick up any weird features immediately.
       If an object is misplaced after the above has been run, something
       is seriously wrong. Either way, this should complain about it
       loudly in the space log. */
    snsSanityCheck(space);
#endif
    
    return;
}

/*
 * Search the range sorted list for the object to insert the specified
 * object after. Objects which are still to be updated this turn are
 * not considered, as they will be updated later.
 *
 * Return value can be NULL, to signify the head of the list, or
 * a pointer to the object in the linked list which should preceed
 * the provided one.
 */
TAG* snsFindInsertPosPrev(TAG *tag, range_t new_range)
{
    TAG *itag;
    
    for (itag = RPREV(tag); itag != NULL; itag = RPREV(itag)) {
	
	if (itag->moveID == move_turn_id)
	    continue;
	
	if (itag->range <= new_range)
	    return itag;
    }
    
    return NULL;
}

/*
 * Search the range sorted list for the object to insert the specified
 * object after. Objects which are still to be updated this turn are
 * not considered, as they will be updated later.
 *
 * Return value can be the provided object, to signify no change,
 * or a pointer to the object in the linked list which should preceed
 * the provided one.
 */
TAG* snsFindInsertPosNext(TAG *tag, range_t new_range)
{
    TAG *itag;
    
    for (itag = tag; RNEXT(itag) != NULL; itag = RNEXT(itag)) {
	
	if (RNEXT(itag)->moveID == move_turn_id)
	    continue;
	
	if (RNEXT(itag)->range >= new_range)
	    return itag;
    }

    return itag;
}

/*
 * Adds the object to the linked list for the given space.
 */
void snsAddObject(TAG *object, int space)
{
    TAG    *tag;
    range_t new_range;
    
    /*
     * Add object to the non-sorted object list fo this space.
     * The object is added at the head of the list.
     */
    object->next = space_info[space].list;
    space_info[space].list = object;
    
    /* Add object to the sorted list at the right place */
    
    object->range = HubRange(object->dc_pos);
    tag = space_info[space].dist;

    if (tag == NULL) {

	/*
	 * If the list was empty, initialize it.
	 */
	space_info[space].dist = object;
	RPREV(object) = NULL;
	RNEXT(object) = NULL;

    } else {
    
	new_range = object->range;

	/*
	 * Search until last object in list or the insertion point found
	 */
	while((tag->range < new_range) && (RNEXT(tag) != NULL))
	    tag = RNEXT(tag);
	
	if (tag->range >= new_range) {
	    
	    /* Insert new object before tag */
	    RPREV(object) = RPREV(tag);
	    RNEXT(object) = tag;
	    
	    if (RPREV(tag) != NULL)
		RNEXT(RPREV(tag)) = object;
	    else
		space_info[space].dist = object;
	    
	    RPREV(tag) = object;
	
	} else {
	    
	    /* Insert new object after tag */
	    RPREV(object) = tag;
	    RNEXT(object) = RNEXT(tag);
	    if (RNEXT(object) != NULL)
		RPREV(RNEXT(object)) = object;
	    
	    RNEXT(tag) = object;
	}
    }
    
    if (Huge(object)) {
	object->huge_prev = NULL;
	object->huge_next = space_info[space].huge;
	
	if (object->huge_next != NULL)
	    object->huge_next->huge_prev = object;
	
	space_info[space].huge = object;
    } else {
	object->huge_prev = NULL;
	object->huge_next = NULL;
    }
    
    return;
}

void snsDelObject(TAG *object, int space)
{
    if (RNEXT(object) != NULL)
	RPREV(RNEXT(object)) = RPREV(object);
    
    if (RPREV(object) != NULL)
	RNEXT(RPREV(object)) = RNEXT(object);
    else
	space_info[space].dist = RNEXT(object);
    
    if (Huge(object)) {
	if (object->huge_next != NULL)
	    object->huge_next->huge_prev = object->huge_prev;
	
	if (object->huge_prev != NULL)
	    object->huge_prev->huge_next = object->huge_next;
	else
	    space_info[space].huge = object->huge_next;
    }
    
    return;
}

void snsCheckSensors(int space)
{
    TAG *entry, *ptr;
    
    /* Reset contacts state for new turn. */
    new_sensor_turn(space);
    
    for (entry=space_info[space].list; entry != NULL; entry=entry->next) {
	
	if (Removed(entry))
	    continue;
	
	if (CanSense(entry)) {
	    
	    /* Check objects forwards in the list */	
	    for (ptr = RNEXT(entry); ptr != NULL; ptr = RNEXT(ptr))
		if (check_pair(entry, ptr))
		    break;
	    
	    /* Check objects backwards in the list */	
	    for (ptr = RPREV(entry); ptr != NULL; ptr = RPREV(ptr))
		if (check_pair(entry, ptr))
		    break;
	    
	    for (ptr=space_info[space].huge; ptr != NULL; ptr=ptr->huge_next)
		if (entry != ptr)
		    sense_huge_object(entry, ptr);
	}
    }
    
    /* Remove stale contacts */
    cleanup_contacts(space);
    
    return;
}

/* check_pair:  Ensure that a full sensor check is only done if sensee is
 *	        within sensor's sensor range.
 *
 * Returns 0 to continue looking in this direction, or 1 to stop.
 */
int check_pair(TAG *senser, TAG *sensee)
{
    if (Invisible(sensee)) {
	update_contact(senser, sensee, NOT_SENSED);
	return 0;
    }
    
    if (Omniscient(senser)) {
	update_contact(senser, sensee, SENSED);
	return 0;
    }
    
    /* Skip huge objects. They're processed separately. */
    if (Huge(sensee)) 
	return 0;
    
    /* See if sensee is even remotely in range */
    if ((RANGEA(sensee->range - senser->range) / 1.5) <= 
	F_INT(senser->rec, sensor_range)) {
	
	snsSenseObject(senser, sensee);
	
	return 0;
    }
    
    /* Done looking in this direction */
    return 1;
}

/* 
 * sense_huge_object:  Use simple formula to see if huge object appears
 * 			on sensors.
 */
void sense_huge_object(TAG *senser, TAG *sensee)
{
    XYZ pos1, pos2;
    int result;
    range_t range;
    struct timeval now;
    
    gettimeofday(&now, NULL);

    object_position(senser, &pos1, &now, 0);
    object_position(sensee, &pos2, &now, 0);

    range = distance(&pos1, &pos2);
    
    if ((range < (F_INT(senser->rec, sensor_range) * 
		  F_FLOAT(sensee->rec, size)) ||
	 Omniscient(senser)) && !Invisible(sensee))
	result = SENSED;	
    else
	result = NOT_SENSED;
    
    update_contact(senser, sensee, result);
    
    return;
}

/*
 * snsSenseObject:  Use more complex formula to see if ship-sized object
 *		    appears on sensors.
 */
void snsSenseObject(TAG *senser, TAG *sensee)
{
    XYZ pos1, pos2, pos3;
    float effective_aspect_ratio; 	/* ease of sensing target */
    float prob;
    float planet_effect, weapons_effect;
    float modified_cloak_eff;
    TAG *hugeobj;
    int i;
    range_t hugerange, range;
    struct timeval now;
    
    gettimeofday(&now, NULL);

    object_position(senser, &pos1, &now, 0);
    object_position(sensee, &pos2, &now, 0);

    range = distance(&pos1, &pos2);
    
    /*
     * If sensee is over three times the sensor range away from the senser,
     * don't bother doing any further calculations (to save time).
     */
    if (range > (3.0 * F_INT(senser->rec, sensor_range))) {
	update_contact(senser, sensee, NOT_SENSED);
	return;
    }
    
    if (Nearsighted(senser)) {
	if (range < F_INT(senser->rec, sensor_range))
	    update_contact(senser, sensee, SENSED);
	else
	    update_contact(senser, sensee, NOT_SENSED);
    }
    
    /*
     * Add cloak effect 
     */

    modified_cloak_eff = 1.0;
    if (Cloaked(sensee)) {
	modified_cloak_eff = F_FLOAT(sensee->rec, cloak_effect) * 10.0;
	
	if (Ship(sensee)) {
	    modified_cloak_eff += 10.0 *
		(100.0 - (float) sensee->shipdata->reactor_setting) /
		(float) sensee->shipdata->reactor_setting;
	}
    }
    
    effective_aspect_ratio = F_FLOAT(sensee->rec, size) / modified_cloak_eff;
    
    /* 
     * Planet effect makes uncloaked ships harder to see, but makes
     * cloaked ships easier to see.  Ignore if less than 5%, however.
     */
    
    hugeobj = nearest_huge_object(senser);
    
    if (hugeobj == NULL)
	planet_effect = 0;
    else {

	object_position(hugeobj, &pos3, &now, 0);

	hugerange= distance(&pos2, &pos3);
	planet_effect = F_FLOAT(hugeobj->rec, size) * 0.80 /
	    (F_FLOAT(hugeobj->rec, size) + 
	     (hugerange * hugerange / 1000000.0));
    }
    
    if (planet_effect >= 0.05) {
	
	if (Cloaked(sensee)) 
	    effective_aspect_ratio *= (1.0 + (2 * planet_effect));
	else
	    effective_aspect_ratio *= (1.0 - planet_effect);
    }
    
    if (Ship(sensee)) {
	
	/*
	 * Weapons effect makes cloaked ships a lot easier to see, and
	 * makes seeing uncloaked ships somewhat easier.
	 */
	
	weapons_effect = (((float)(sensee->shipdata->num_torps_online)
			   *TORP_CHARGING_PENALTY) + 
			  ((float)(sensee->shipdata->num_guns_online) *
			   GUN_ONLINE_PENALTY));
	
	/* count armed torps */
	for (i=0; i < F_INT(sensee->shipdata->rec, torp_qty); i++)
	       if (sensee->shipdata->torp[i].status == TORP_ARMED)
		   weapons_effect += TORP_ARMED_PENALTY;
    }
    else 
	weapons_effect = 0.0;
    
    if (Cloaked(sensee))
	effective_aspect_ratio += weapons_effect;
    else
	effective_aspect_ratio += weapons_effect * 0.25;
    
     if (Hazy(sensee))
	 effective_aspect_ratio *= 0.3;
     
     if (HasCataracts(senser))
	 effective_aspect_ratio *= 0.3;
     
     /* If sensee is locked on sensor, he's really easy to see. */
     if (sensee->locked_on != NULL)
	 if (sensee->locked_on->listref == senser)
	     effective_aspect_ratio += 10.0;
     
     if (sensee->pending_lock != NULL)
	 if (sensee->pending_lock->listref == senser)
	     effective_aspect_ratio += 10.0;
     
     /* 
      * If sensee has engaged a tractor beam on the senser, or vice-versa
      * he's also really easy to see.
      */
     
     if (Ship(senser) && senser->shipdata->tractor_target == sensee)
	 effective_aspect_ratio += 10.0;
     
     if (senser->tractor_source == sensee)
	 effective_aspect_ratio += 10.0;
     
     prob = 1.0 - range / (F_INT(senser->rec, sensor_range) * 
			   effective_aspect_ratio);
     
     /* Warp speed adjustment */
     prob += ((sensee->speed - senser->speed) / 50.0);
     
     /* Bonus if senser has already made contact with sensee */
     if (snsFindContact(senser, sensee) != NULL)
	 prob += 0.25;
     
     if (senser->locked_on != NULL)
	 if (senser->locked_on->listref == sensee)
	     prob += 0.25;
     
     if (FRAND < prob)
	 update_contact(senser, sensee, SENSED);
     else
	 update_contact(senser, sensee, NOT_SENSED);
     
     return;
}

/*
 * nearest_huge_object:   Find the nearest 'huge' object.
 */
TAG *nearest_huge_object(TAG *senser)			
{
    XYZ pos1, pos2;
    TAG *nearobj = NULL;
    range_t nearrange, range;
    CONTACT *cptr;
    struct timeval now;
    
    nearrange = MAX_RANGE;
    
    gettimeofday(&now, NULL);

    object_position(senser, &pos1, &now, 0);

    for (cptr=senser->contact_list;cptr != NULL; cptr=cptr->next) {
	if (!Huge(cptr->listref))
	    continue;

	object_position(cptr->listref, &pos2, &now, 0);

	range = distance(&pos1, &pos2);
	if (range < nearrange || nearobj==NULL) {
	    nearobj = cptr->listref;
	    nearrange = range;
	}
    }
    
    return nearobj;
}

/* Contact handling routines */

/*
 * new_sensor_turn:  Set all the 'updated' flags to FALSE on each contact
 *                   by incrementing the sensor_turn_id parameter.
 */
void new_sensor_turn(int space)
{
    sensor_turn_id++;
    
    return;
}

/* cleanup_contacts:  Remove contacts that haven't been updated this turn.
 *			Assume that they no longer exist. 
 */
void cleanup_contacts(int space)
{
    TAG *ptr;
    CONTACT *cptr, *cprev, *tmpptr;
    
    for (ptr=space_info[space].list; ptr != NULL; ptr=ptr->next) {
	
	cprev = NULL;
	cptr = ptr->contact_list;
	
	while (cptr != NULL) {
	    
	    if (cptr->updated != sensor_turn_id) {
		
		contact_lost(ptr, cptr, 0);
		if (cprev == NULL) 
		    ptr->contact_list = cptr->next;
		else
		    cprev->next = cptr->next;
		tmpptr = cptr;
		cptr = cptr->next;
		pse_free(tmpptr);
	    }
	    else {
		cprev = cptr;
		cptr = cptr->next;
	    }
	}
    }
    
    return;
    
}

/* snsRemoveContact:  Remove sensee.  */
void snsRemoveContact(TAG *senser, TAG *sensee, int silent)
{
    
    CONTACT *cprev = NULL;
    CONTACT *cptr = senser->contact_list;
    CONTACT *tmpptr;
    
    while (cptr != NULL) {
	
	if (cptr->listref == sensee) {
	    
	    contact_lost(senser, cptr, silent);
	    
	    if (cprev == NULL)
		senser->contact_list = cptr->next;
	    else
		cprev->next = cptr->next;
	    
	    tmpptr = cptr;
	    cptr = cptr->next;
	    pse_free(tmpptr);
	}
	else {	
	    cprev = cptr;
	    cptr = cptr->next;
	}
    }
    
    return;
}

void update_contact(TAG *senser, TAG *sensee, int action)
{
    XYZ pos;
    CONTACT *contact;
    float prob, roll;
    int new_info_level, is_new_contact = 0;
    struct timeval now;

    gettimeofday(&now, NULL);
    
    contact = snsFindContact(senser, sensee);
    
    if (action == SENSED) {
	
	if (contact == NULL) {
	    contact = add_contact(senser, sensee);
	    is_new_contact = 1;
	}
	
	if (contact->turns_of_contact < 65536)
	    contact->turns_of_contact++;
	
	object_position(contact->listref, &pos, &now, 0);
	
	contact->turns_since_last_contact = 0;
	contact->last_pos.x = pos.x;
	contact->last_pos.y = pos.y;
	contact->last_pos.z = pos.z;
	contact->updated = sensor_turn_id;
	
	if (contact->info_level == 5)
	    return;
	
	/* See how much information we can get. */
	prob = (float)(contact->turns_of_contact) / 20.0;
	roll = FRAND;
	
	if (Omniscient(senser) || Nearsighted(senser) || Huge(sensee))
	    roll = 0;
	
	if (roll < prob)
	    new_info_level = 5;
	else if (roll < (prob + .10))
	    new_info_level = 4;	
	else if (roll < (prob + .20))
	    new_info_level = 3;	
	else if (roll < (prob + .30))
	    new_info_level = 2;
	else
	    new_info_level = 1;
	
	/* Never show full info on cloakers */	
	if (Cloaked(contact->listref) && new_info_level > 3)
	    new_info_level = 3;
	
	if (is_new_contact)
	    new_contact(senser, contact);
	
	if (new_info_level > contact->info_level) {
	    contact->info_level = new_info_level;
	    if (!is_new_contact && Ship(senser))
		snsDisplaySensorInfo(senser, contact, -1, DSD_UPDATE);
	}
	
    } else {
	
	if (contact == NULL)
	    return;			
	
	contact->turns_since_last_contact++;
	
	if (!Cloaked(contact->listref) && !Invisible(contact->listref)
	    && !Huge(contact->listref) && 
	    contact->turns_since_last_contact < 3)
	    contact->updated = sensor_turn_id;
    }
    
    return;
}

/*
 * add_contact:  Add contact to contact list
 */
CONTACT *add_contact(TAG *source, TAG *contact)
{
    
    CONTACT *ptr;
    CONTACT *newcontact;
    
    newcontact = (CONTACT *) pse_malloc(sizeof(CONTACT));
    
    newcontact->listref = contact;
    newcontact->turns_of_contact = 0;
    newcontact->turns_since_last_contact = 0;
    newcontact->info_level = 0;
    newcontact->last_pos.x = 0;
    newcontact->last_pos.y = 0;
    newcontact->last_pos.z = 0;
    newcontact->watcher = 0;
    newcontact->inside_critical = FALSE;
    newcontact->next = NULL;
    
    if (source->contact_list==NULL) {
	
	source->contact_list = newcontact;
	source->contact_offset = 1;
	
    } else {
	for (ptr=source->contact_list; ptr->next != NULL; ptr=ptr->next)
	    ;
	ptr->next = newcontact;
	source->contact_offset++;
    }
    
    newcontact->contact_number = source->contact_offset;
    
    return(newcontact);
}

/*
 * snsFindContact:  Find contact in contact list
 */
CONTACT *snsFindContact(TAG *source, TAG *contact)
{
    CONTACT *ptr;
    
    for (ptr=source->contact_list; ptr != NULL; ptr=ptr->next) 
	if (ptr->listref==contact)
	    break;
    
    return(ptr);
}

CONTACT *snsFindContactByNumber(TAG *source, int num)
{
    CONTACT *ptr;
    
    for (ptr=source->contact_list; ptr != NULL; ptr=ptr->next) 
	if (ptr->contact_number == num)
	    break;
    
    return(ptr);
}

void new_contact(TAG *senser, CONTACT *contact)
{
    if (Ship(senser))
	snsDisplaySensorInfo(senser, contact, -1, DSD_NEW);
    
    if (EventDriven(senser)) 
	evTrigger(senser, EVENT_NEW_CONTACT, "c", contact);
    
#ifdef LOG_CONTACTS
    if (space_info[senser->space].flags & SPACE_LOGGED) {

	XYZ pos1, pos2;
	struct timeval now;
	
	gettimeofday(&now, NULL);

	object_position(senser, &pos1, &now, 0);
	object_position(contact->listref, &pos2, &now, 0);

	log_space("NEW CONTACT!  %s (#%d) sees %s (#%d) at range " RANGEF ".",
		  senser->name, senser->data_object, contact->listref->name, 
		  contact->listref->data_object,
		  distance(&pos1, &pos2));
    }
#endif
}

/*
 * snsListContacts:  Spit contact list into buff, as either contact #'s 
 *		     (format = 0), or dbrefs (format = 1).  Show the entire
 *		     list (type=0), objects that can be beamed to (type=1),
 *		     and objects that can be beamed (type=2).
 */
void snsListContacts(char *buff, CONTACT *ptr, int type, int format)
{
    static char *buffstart = NULL;
    
    if (buffstart == NULL) {
	buffstart = buff;
	buff[0]='\0';
    }
    
    if ((buff - buffstart >= LARGE_BUF_SIZE - 10) || (ptr == NULL)) {
	if (buff[strlen(buff) - 1] == ' ')
	    buff[strlen(buff) - 1] = '\0';
	buffstart = NULL;
	return;
    }
    
    if ((type==0) || (type==1 && CanBeamTo(ptr->listref)) ||
	((type==2) && Beamable(ptr->listref))) {
	
	if (format)
	    sprintf(buff, "#%d ", ptr->listref->rec->db_obj);
	else
	    sprintf(buff, "%d ", ptr->contact_number);
	
    }
    
    snsListContacts(buff + strlen(buff), ptr->next, type, format);
    
    buffstart = NULL;
    return;
}

void snsDisplayContacts(TAG *object, void *mud, int show_ships, int show_other)
{
    SSTR *ps;
    CONTACT *contact;
    int found = 0;
    struct timeval now;
    
    gettimeofday(&now, NULL);

    ps = sstr_new(MAX_ATTRIBUTE_LEN);
    
    for (contact=object->contact_list; contact != NULL;
	 contact=contact->next) {
	
	if (Ship(contact->listref) && !show_ships) continue;
	if (!Ship(contact->listref) && !show_other) continue;
	
	snsFormatShortSensorInfo(object, contact, ps, &now);
	found++;
    }
    
    if (!found)
	WriteFunctionBuffer("No contacts on sensors.");
    else
	WriteFunctionBuffer(sstr_str(ps));
    
    sstr_free(ps);
    
    return;
}

/*
 * Called by ship objects for new and updated contacts, and current contacts.
 * Called by non-ship objects current contacts only.
 */

void snsDisplaySensorInfo(TAG *senser, CONTACT *contact, dbref player, 
			  int control)
{
    XYZ pos1, pos2;
    char buff[SMALL_BUF_SIZE];
    XYZ xyz;
    SPH sph;
    TAG *target;
    struct timeval now;
    
    /* Do some sanity checking. If it's not a ship and player is -1, we don't
       know where to emit messages to. */
    if ((player == -1) && !Ship(senser))
	return;

    target = contact->listref;
    
    gettimeofday(&now, NULL);

    object_position(senser, &pos1, &now, 0);
    object_position(target, &pos2, &now, 0);
    
    /* calculate relative positions */
    xyz.x = pos2.x - pos1.x;
    xyz.y = pos2.y - pos1.y;
    xyz.z = pos2.z - pos1.z;
    xyz_to_sph(xyz, &sph);
    
    /* display header message */
    switch (control) {
	
    case DSD_NEW:
	
	/* New contacts are only reported for ship objects */
	/* force the update to all tac consoles */
	player = -1;
	
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE,
		 "NEW CONTACT - designated number %d",
		 contact->contact_number);
	
	break;
	
    case DSD_UPDATE:
	
	/* Updated contacts are only reported for ship objects */
	/* force the update to all tac consoles */
	player = -1;
	
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE,
		 "Update:  Contact [%d] further identified as:",
		 contact->contact_number);
	break;
	
    case DSD_ROUTINE:
	
	/* Add a [*] if the target is locked onto us */
	if ((contact->listref->locked_on != NULL) &&
	    (contact->listref->locked_on->listref == senser)) 
	    sprintf(buff, "[*] Contact [%d]:", contact->contact_number);
	else
	    sprintf(buff, "Contact [%d]:", contact->contact_number);
	
	if (player == -1)
	    MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "%s",
		     buff);
	else
	    Notify(player, buff);
	
	break;
    }
    
    build_contact_name(buff, contact, SMALL_BUF_SIZE);
    if (player == -1)
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "%s", buff);
    else
	Notify(player, buff);
    
    sprintf(buff, "   Contact bearing %3.2f elevation %+2.2f range "
	    RANGEF_NP " %s",
	    sph.bearing, sph.elevation, sph.range * 
	    F_FLOAT(senser->rec, range_factor), 
	    F_PTR(senser->rec, range_name));
    
    if (player == -1)
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "%s", buff);
    else
	Notify(player, buff);
    
    if (contact->info_level > 1)  {
	
	if (Ship(target) || target->speed > 0) {
	    /* construct the target's movement string */
	    if (player == -1)
		MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE,
			 "   Contact heading %3.2f%+2.2f at "
			 "warp %3.1f", target->heading.bearing,
			 target->heading.elevation,
			 target->speed);
	    else
		FNotify(player, "   Contact heading %3.2f%+2.2f at "
			"warp %3.1f", target->heading.bearing, 
			target->heading.elevation, 
			target->speed);
	}
	
	if (Ship(senser)) {
	    if (Ship(target))
		/* and give the shield facings */
		sprintf(buff, "   Our %s side is facing their %s side.",
			shdFullName(senser,
				    calcFacingShield(pos2, senser)),
			shdFullName(target,
				    calcFacingShield(pos1, target)));
	    else 
		sprintf(buff, "   Our %s side is facing the contact.",
			shdFullName(senser,
				    calcFacingShield(pos2,senser)));
	    
	    if (player == -1)
		MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "%s",
			 buff);
	    else
		Notify(player, buff);
	}
    }
    
    if (Cloaked(target)) {
	if (player == -1)
	    MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE,
		     "Contact is currently cloaked.");
	else
	    Notify(player, "Contact is currently cloaked.");
    }
    
    if (player == -1)
	MFNotify(senser->shipdata, player, CONS_TAC, CONS_ACTIVE, "");
    else
	Notify(player, "");
    
    return;
}

void snsDisplayShortContacts(TAG *object, void *mud, int show_ships,
			     int show_other)
{
    SSTR *ps;
    CONTACT *contact;
    int found = 0;
    struct timeval now;

    gettimeofday(&now, NULL);
    
    ps = sstr_new(MAX_ATTRIBUTE_LEN);
    
    sstr_cat(ps,
	     "##### Name                   Bearing       "
	     "Range   Heading       Speed Shields\n"
	     "----- ---------------------- ------------- "
	     "------- ------------- ----- -------\n", 0);
    
    for (contact=object->contact_list; contact != NULL;
	 contact=contact->next) {
	if (Ship(contact->listref) && !show_ships) continue;
	if (!Ship(contact->listref) && !show_other) continue;
	
	if (Removed(contact->listref)) continue;
	
	snsFormatShortSensorInfo(object, contact, ps, &now);
	
	found ++;
    }
    
    if (!found)
	WriteFunctionBuffer("No contacts on sensors.");
    else
	WriteFunctionBuffer(sstr_str(ps));

    sstr_free(ps);
    
    return;
}

void snsFormatShortSensorInfo(TAG *senser, CONTACT *contact, SSTR *ps,
			      struct timeval *now)
{
    XYZ pos1, pos2;
    char name[SMALL_BUF_SIZE], range[SMALL_BUF_SIZE], heading[SMALL_BUF_SIZE];
    const char *senser_shield;
    const char *target_shield;
    char locked;
    
    XYZ xyz;
    SPH sph;
    TAG *target;
    
    target = contact->listref;
    
    object_position(senser, &pos1, now, 0);
    object_position(target, &pos2, now, 0);

    /* calculate relative positions */
    xyz.x = pos2.x - pos1.x;
    xyz.y = pos2.y - pos1.y;
    xyz.z = pos2.z - pos1.z;
    xyz_to_sph(xyz, &sph);
    
    if ((contact->listref->locked_on != NULL) &&
	(contact->listref->locked_on->listref == senser))
	locked = '*';
    else
	locked = ' ';
    
    build_contact_name(name, contact, SMALL_BUF_SIZE);
    
    if (Ship(senser))
	senser_shield = shdCharName(senser,
				    calcFacingShield(pos2,senser));
    else
	senser_shield = " ";
    
    if (Ship(target))
	target_shield = shdCharName(target,
				    calcFacingShield(pos1,target));
    else
	target_shield = " ";
    
    navRangeString(senser, sph.range, range, 7);
    
    if (contact->info_level == 0)
	strcpy(heading, "---.-- --.-- ---.- ");
    else
	sprintf(heading, "%6.2f %+6.2f %+5.1f ", target->heading.bearing,
		target->heading.elevation, target->speed);
    
    sstr_cat_sprintf(ps, "%c%4d %-22.22s %6.2f %+6.2f %7.7s %19.19s  "
		     "%2.2s %2.2s\n",
		     locked, contact->contact_number, name,
		     sph.bearing, sph.elevation, range,
		     heading, senser_shield, target_shield);
    
    return;
}

void build_contact_name(char *name, CONTACT *contact, int buffsize)
{

     char *fargs[1];
     SHIP *ship;
     TAG *target;
     char *evbuff;

     target = contact->listref;
     ship = target->shipdata;

     fargs[0]=AllocBuffer("evDisplay");
     sprintf(fargs[0], "%d", contact->info_level);
     
     /* Allocate the evaluation buffer and evalute the attribute */
     evbuff = pse_malloc(MAX_ATTRIBUTE_LEN);
     getEvalAttrBuf(target->rec->db_obj, INFO_CONTACT_STRING,fargs,1,evbuff);

     /* Copy the result into the name buffer and free the eval buffer */
     strncpy(name, evbuff, buffsize-1);
     name[buffsize-1] = '\0';
     pse_free(evbuff);
     
     FreeBuffer(fargs[0]);

     if (strlen(name)==0)
	  strcpy(name, target->rec->name);
     else
          return;

     if (!Ship(target))
	  return;

     switch (contact->info_level) {

     case 5:
	  sprintf(name, "%s -- %s %s-class %s", target->rec->name,
		  F_PTR(ship->rec, owner_name), F_PTR(ship->rec, shipclass), 
		  F_PTR(ship->rec, type));

	  break;

     case 4:
	  sprintf(name, "%s %s-class %s", F_PTR(ship->rec, owner_name),
		  F_PTR(ship->rec, shipclass), F_PTR(ship->rec, type));
	  break;
     case 3:
	  sprintf(name, "%s %s", F_PTR(ship->rec, owner_name), 
		  F_PTR(ship->rec, type));
	  break;
     default:
	  sprintf(name, "%s", F_PTR(ship->rec, type));
	  break;
     }
}

void snsContactString(TAG *senser, CONTACT *contact, char *buff,
		      struct timeval *now)
{
    XYZ pos1, pos2;
    char name[SMALL_BUF_SIZE], head[SMALL_BUF_SIZE];
    const char *senser_shield;
    const char *target_shield;
    
    TAG *target;
    XYZ xyz;
    SPH sph;
    
    target = contact->listref;

    object_position(senser, &pos1, now, 0);
    object_position(target, &pos2, now, 0);
    
    /* calculate relative positions */
    xyz.x = pos2.x - pos1.x;
    xyz.y = pos2.y - pos1.y;
    xyz.z = pos2.z - pos1.z;
    xyz_to_sph(xyz, &sph);
    
    /* Work out the name of the contact based on visibility etc */
    build_contact_name(name, contact, SMALL_BUF_SIZE);

    if (contact->info_level == 0)
	strcpy(head, "? +? ?");
    else
	sprintf(head, "%.2f %+.2f %.1f", target->heading.bearing,
		target->heading.elevation, target->speed);
    
    senser_shield = shdCharName(senser,calcFacingShield(pos2,senser));
    target_shield = shdCharName(target,calcFacingShield(pos1,target));
    
    if (!Ship(target) || (contact->info_level == 0))
	target_shield = "-";
    
    sprintf(buff, "SIZE: %.2f VISBILITY: %d BEARING: %.2f %+.2f %f "
	    "HEADING: %s FACING: %2.2s %2.2s NAME: %s",
	    F_FLOAT(target->rec, size), contact->info_level,
	    sph.bearing, sph.elevation, sph.range * 
	    F_INT(senser->rec, range_factor),
	    head, senser_shield, target_shield, name);
}

void snsScan(TAG *object, void *mud, int num)
{
    XYZ xyz, pos1, pos2;
    SPH sph;
    SHIP *scannee;
    CONTACT *source;
    CONTACT *target;
    char *fargs[1];
    char *rb;
    SSTR *ps;
    struct timeval now;
    
    if (object->shipdata->damage[SYSTEM_SCANNERS].status > '6') {
	WriteFunctionBuffer("Scanners are damaged and inoperable.");
	return;
    }
    
    target = snsFindContactByNumber(object, num);
    
    if (target==NULL) {
	WriteFunctionBuffer("Invalid target.");
	return;
    }
    
    gettimeofday(&now, NULL);

    object_position(object, &pos1, &now, 0);
    object_position(target->listref, &pos2, &now, 0);

    if (distance(&pos1, &pos2) > 
	F_INT(object->shipdata->rec, scanner_range)) {
	WriteFunctionBuffer("That object is out of scanner range.");
	return;
    }
    
    /* Raise our info level on the target object. */
    if (Cloaked(target->listref))
	target->info_level = Max(target->info_level, 4);
    else
	target->info_level = 5;
    
    ps = sstr_new(MAX_ATTRIBUTE_LEN);
    
    sstr_sprintf(ps, "Scanning [%d]: %s\n", target->contact_number,
		 target->listref->rec->name);
    
    xyz.x = pos2.x - pos1.x;
    xyz.y = pos2.y - pos1.y;
    xyz.z = pos2.z - pos1.z;
    xyz_to_sph(xyz, &sph);
    
    /*
     * Scanning info strings always relies on the value from the evaluation.
     * If you need it to emulate the old behaviour, just return the desc from
     * the eval.
     */
    fargs[0]=AllocBuffer("snsScan");
    sprintf(fargs[0], "%d", target->contact_number);
    
    rb = pse_malloc(MAX_ATTRIBUTE_LEN);
    getEvalAttrBuf(object->rec->db_obj, INFO_SCAN_STRING, fargs, 1, rb);
    if (*rb != '\0')
	sstr_cat(ps, rb, 0);
    
    pse_free(rb);
    FreeBuffer(fargs[0]);
    
    /* We're done for non-ship objects */
    if (!Ship(target->listref)) {
	WriteFunctionBuffer(sstr_str(ps));
	sstr_free(ps);
	return;
    }
    
    scannee = target->listref->shipdata;
    
    sstr_cat_sprintf(ps, "Target bearing %3.2f%+2.2f at range %.f %s\n",
		     sph.bearing, sph.elevation, sph.range * 
		     F_FLOAT(object->rec, range_factor), 
		     F_PTR(object->rec, range_name));
    
    sstr_cat_sprintf(ps, "       heading %3.2f%+2.2f at warp %1.0f\n",
		     target->listref->heading.bearing,
		     target->listref->heading.elevation,
		     target->listref->speed);	
    
    /* weapons status */
    
    sstr_cat(ps, "Weapons:\n", 0);
    sstr_cat_sprintf(ps, "    %d %ss online -- energy allocated %d\n",
		     scannee->num_guns_online, F_PTR(scannee->rec, gun_string),
		     scannee->talloc_guns);
    sstr_cat_sprintf(ps, "    %d %ss online -- energy allocated %d\n\n",
		     scannee->num_torps_online,
		     F_PTR(scannee->rec, torp_string),
		     scannee->talloc_torps);
    
    if (Cloaked(target->listref))
	sstr_cat(ps, "Target's cloaking device is engaged.\n", 0);
    else {
	sstr_cat(ps, "Shields:\n", 0);
	
#ifdef ENABLE_SHIELD_CLASSES
	sstr_cat(ps, "*FIX ME*\n", 0);
#else
	switch(scannee->shield_number) {
	case 4:
	    sstr_cat_sprintf(ps, "%s: %d:%s  %s: %d:%s  %s: %d:%s  %s: %d:%s\n",
			     shdShortName(target->listref, 0),
			     scannee->shield_level[0], Shield_string(scannee, 0),
			     shdShortName(target->listref, 1),
			     scannee->shield_level[1], Shield_string(scannee, 1),
			     shdShortName(target->listref, 2),
			     scannee->shield_level[2], Shield_string(scannee, 2),
			     shdShortName(target->listref, 3),
			     scannee->shield_level[3], Shield_string(scannee, 3));
	    break;
	case 6:
	    sstr_cat_sprintf(ps, "%s: %d:%s  %s: %d:%s  %s: %d:%s  %s: %d:%s  "
			     "%s: %d:%s  %s: %d:%s\n",
			     shdShortName(target->listref, 0),
			     scannee->shield_level[0], Shield_string(scannee, 0),
			     shdShortName(target->listref, 1),
			     scannee->shield_level[1], Shield_string(scannee, 1),
			     shdShortName(target->listref, 2),
			     scannee->shield_level[2], Shield_string(scannee, 2),
			     shdShortName(target->listref, 3),
			     scannee->shield_level[3], Shield_string(scannee, 3),
			     shdShortName(target->listref, 4),
			     scannee->shield_level[4], Shield_string(scannee, 4),
			     shdShortName(target->listref, 5),
			     scannee->shield_level[5], Shield_string(scannee, 5));
	    break;
	case 1:
	    sstr_cat_sprintf(ps, "%s: %d:%s\n",
			     shdShortName(target->listref, 0),
			     scannee->shield_level[0], Shield_string(scannee, 0));
	    break;
	case 0:
	    sstr_cat(ps, "Target is not shielded.\n", 0);
	    break;
	default:
	    sstr_cat_sprintf(ps, "Illegal number of shields: %d\n",
			     scannee->shield_number);
	    break;
	}
#endif
    }
    
#ifndef ENABLE_SHIELD_CLASSES
    if (scannee->shield_number > 0)
	sstr_cat_sprintf(ps, "We are currently facing their %s shield.\n",
			 shdFullName(target->listref,
				     calcFacingShield(pos1,
						      target->listref)));
    
    if (object->shipdata->shield_number > 0)
	sstr_cat_sprintf(ps, "They are currently facing our %s shield.\n",
			 shdFullName(object,
				     calcFacingShield(pos2,
						      object)));
#endif
    
    sstr_cat(ps, "\n", 0);
    
    if (scannee->door_status != DOORS_NONE) {
	sstr_cat_sprintf(ps, "The target's cargo bay doors are %s.\n\n",
			 (scannee->door_status == DOORS_CLOSED) ?
			 "closed" : "open");
    }
    
    sstr_cat_sprintf(ps, "Hull integrity:  %2.0f%%\tpower to warp: %d\n",
		     (float)scannee->current_integrity / 
		     (float)F_INT(scannee->rec, hull) * 100.0,
		     scannee->alloc_nav);

    /* Copy the results into the function return buffer */
    WriteFunctionBuffer(sstr_str(ps));
    sstr_free(ps);
    
    /* Warn the scanee that they are being scanned */
    source = snsFindContact(target->listref, object);
    
    if (source != NULL)
	MFNotify(scannee, -1, CONS_TAC, CONS_ACTIVE,
		 "We are being scanned by contact [%d]: %s.",
		 source->contact_number, object->rec->name);
    else
	MFNotify(scannee, -1, CONS_TAC, CONS_ACTIVE,
		 "We are being scanned by an unknown source.");
    
    return;
}

void contact_lost(TAG *senser, CONTACT *contact, int silent)
{
    char buff[SMALL_BUF_SIZE];
    XYZ last_position, pos1, pos2;
    SPH last_relative;
    struct timeval now;

    gettimeofday(&now, NULL);

    object_position(senser, &pos1, &now, 0);
    object_position(contact->listref, &pos2, &now, 0);
    
    if (Ship(senser) && !silent) {
	if (F_INT(senser->shipdata->rec, hull) < 0)
	    return;
	
	MFNotify(senser->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		 "CONTACT [%d] LOST:",contact->contact_number);
	build_contact_name(buff, contact, SMALL_BUF_SIZE);
	MFNotify(senser->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		 "%s", buff);
	
	last_position.x = contact->last_pos.x - pos1.x;
	last_position.y = contact->last_pos.y - pos1.y;
	last_position.z = contact->last_pos.z - pos1.z;
	
	xyz_to_sph(last_position, &last_relative);
	
	MFNotify(senser->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		 "Contact last spotted bearing: "
		 "%3.0f%+2.0f, range %.f %s", last_relative.bearing,
		 last_relative.elevation,
		 last_relative.range * F_FLOAT(senser->rec, range_factor),
		 F_PTR(senser->rec, range_name));
	
	if (contact->info_level > 1) 
	    MFNotify(senser->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		     "                     heading: %3.0f"
		     "%+2.0f at warp %3.1f", 
		     contact->listref->heading.bearing,
		     contact->listref->heading.elevation, 
		     contact->listref->speed);
    }
    
    if (senser->locked_on == contact)
	tacBreakLock(senser);
    
    if (Ship(senser))
	if (senser->shipdata->tractor_target == contact->listref)
	    tacBreakTractor(senser);
    
    /*
     * If the lock is broken before it is achieved, we can simulate a
     * message to send to the tacofficer, by granting the lock then breaking
     * it.
     */
    
    if (senser->pending_lock == contact) {
	senser->locked_on = contact;
	tacBreakLock(senser);
	senser->pending_lock = NULL;
    }
    
    if (EventDriven(senser) && !silent)  {
	
	if (contact->inside_critical)
	    evTrigger(senser, EVENT_OUTSIDE_CRITICAL, "c", contact);
	
	evTrigger(senser, EVENT_CONTACT_LOST, "c", contact);
    }
    
#ifdef LOG_CONTACTS
    if ((space_info[senser->space].flags & SPACE_LOGGED) && !silent)
	log_space("CONTACT LOST.  %s (#%d) no longer sees %s (#%d), "
		  "range " RANGEF ".", senser->name, senser->data_object, 
		  contact->listref->name, contact->listref->data_object,
		  distance(&pos2, &pos1));
#endif
    
    return;
}

#ifdef ENABLE_DEBUGGING
/*
 * Do some diagnostics on the sensor related information.
 */
void snsSanityCheck(int space)
{
    CONTACT *ct;
    TAG     *tag, *dtag;
    int     tc, dc;
    
    /* Count the number of objects in space */
    for (tag = space_info[space].list, tc=0; tag != NULL; tag = tag->next)
	tc++;
    
    /* Count the number of objects on the range list */
    for (tag = space_info[space].dist, dc=0; tag != NULL; tag = RNEXT(tag))
	dc ++;
    
    if (tc != dc)
	log_space("Insanity %d: tc(%d) != dc(%d)", space, tc, dc);
    
    /* Check that all objects on the space list are on the range list */
    for (tag = space_info[space].list; tag != NULL; tag = tag->next) {
	for (dtag = space_info[space].dist; (dtag != tag) && (dtag != NULL);
	     dtag = RNEXT(dtag))
	    ;
	
	if (!dtag)
	    log_space("Insanity %d: Object %d not on distance list.",
		      space, tag->rec->db_obj);
    }
    
    /* Check that all objects on the range list are on the object list */
    for (tag = space_info[space].dist; tag != NULL; tag = RNEXT(tag)) {
	for (dtag = space_info[space].list; (dtag != tag) && (dtag != NULL);
	     dtag = dtag->next)
	    ;
	
	if (!dtag)
	    log_space("Insanity %d: Object %d not on space object list.",
		      space, tag->rec->db_obj);
    }
    
    /* Check for duplicates */
    for (tag = space_info[space].list; tag != NULL; tag = tag->next)
	for (dtag = tag->next; dtag != NULL; dtag = dtag->next)
	    if (dtag->rec->db_obj == tag->rec->db_obj)
		log_space("Insanity %d: Object %d in space list twice.",
			  space, tag->rec->db_obj);
    
    /* Check for duplicates */
    for (tag = space_info[space].dist; tag != NULL; tag = RNEXT(tag))
	for (dtag = RNEXT(tag); dtag != NULL; dtag = RNEXT(dtag))
	    if (dtag->rec->db_obj == tag->rec->db_obj)
		log_space("Insanity %d: Object %d in distance list twice.",
			  space, tag->rec->db_obj);
    
    /* Check for incorrectly sorted range list */
    for (tag = space_info[space].dist; tag != NULL; tag = RNEXT(tag))
	if ((tag->range_next) && (RNEXT(tag)->range < tag->range)) {
	    log_space("Insanity %d (%d): Objects %d (%d:%d) / %d (%d:%d) "
		      "misplaced in range list.", space, move_turn_id,
		      tag->rec->db_obj,
		      tag->moveID,
		      tag->range,
		      RNEXT(tag)->rec->db_obj,
		      RNEXT(tag)->moveID,
		      RNEXT(tag)->range);
	    /*
	     * These should automagically fix the error next sensor recalc
	     * though they are not enabled by default because if there is
	     * an error, we need to know.
	     */
#if 0
	    tag->moveID = move_turn_id + 1;
	    tag->range_next->moveID = move_turn_id + 1;
#endif
	}
    
    /* Check the contact list for empty entries */
    for (tag = space_info[space].list; tag != NULL; tag=tag->next) {
	for (ct = tag->contact_list; ct != NULL; ct=ct->next) {
	    if (ct->info_level == 0)
		log_space("Insanity %d: Object %d losing contact with %d",
			  space, tag->rec->db_obj, ct->listref->rec->db_obj);
	}
    }
}
#endif
