/* v1.1
 *
 * scm.h:  Shield Class Module header.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 *
 * Plugin modules provide a set of interface functions in the
 * SCM_INTERFACE structure.
 *
 * A set of defined geometries are stored in the SCAREAS structures,
 * and they define a number of regions.
 *
 * Each ship allocates a number of SCDATA structures, which contain the
 * data for that particular layer.
 */

#ifndef __SCM_H__
#define __SCM_H__

#define SCM_MAX_LAYERS          4    /* Max layers of shielding */
#define SCM_MAX_AREAS           8    /* Max distinct areas per layer */
#define SCM_MAX_NAME_LEN        80   /* Max length of the layer name */

typedef struct scm_interface    SCM_INTERFACE; /* Code entry points */
typedef struct sc_areas         SCAREAS;       /* Area definitions  */
typedef struct sc_data          SCDATA;        /* Per-object data   */

/* Interface function definitions */
typedef int (*scmInitModule_t)(void);
typedef int (*scmFreeModule_t)(void);
typedef int (*scmLoadLayer_t)(dbref, SCDATA*, const char *);
typedef int (*scmSaveLayer_t)(SSTR *, SCDATA*);
typedef int (*scmFreeLayer_t)(SCDATA*);

struct scm_interface {
    char            scmName[SCM_MAX_NAME_LEN];  /* Unique module identifier */
    int             scmNameLen;                 /* Length of name */
    scmInitModule_t scmInitModule;              /* Functions */
    scmFreeModule_t scmFreeModule;
    scmLoadLayer_t  scmLoadLayer;
    scmSaveLayer_t  scmSaveLayer;
    scmFreeLayer_t  scmFreeLayer;
};

struct sc_areas {
    int           areas;                        /* Number of areas        */
    int           layout[SCM_MAX_AREAS][7];     /* Geometry definition    */
    void          *storage;                     /* Layer global storage   */
    int           attrib;                       /* Data Attribute ID      */
};

struct sc_data {
    char          name[SCM_MAX_NAME_LEN];       /* Name the user sees     */
    SCM_INTERFACE *plugin;                      /* Interface data         */
    SCAREAS       *geometry;                    /* Geometry data          */
    void          *storage;                     /* Layer storage          */
    int           attrib;                       /* Data Attribute ID      */
};

#ifdef DEBUG
extern int scAssertValid(SCDEF *);
#define SCASSERT(x) if(scAssertValid(x) == 0) return "OOPS: Invalid shield class pointer.";
#else
#define SCASSERT(x)
#endif

extern void scmInitModules(void);
extern const char *scmListRegisteredModules(void);
extern int scmLoadData(dbref, SCDATA*, const char *);
extern int scmSaveData(dbref, SCDATA*);
extern int scmFreeData(SCDATA*);

/* SCM internal interfaces */
const char *scmParseData(const char *data, int *rel, SCAREAS **area);

#endif
