/* v1.1
 *
 * class.c:  Class/Record Module Code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "spaceconf.h"
#include "pseint.h"
#include "btree.h"
#include "space.h"
#include "class.h"
#include "object.h"
#include "smisc.h"

Class *class_head;
Class *class_tail;

BNODE *class_names;

Record *load_record(Class *clptr, dbref record);
void free_record(Record *record);

const char bad_record[] = "The object has not been set up properly.";

void clsInit()
{
    class_head = class_tail = NULL;
    class_names = 0;
}

void clsShutdown()
{
    Class *ptr, *tmp;

    btreeFree(class_names);

    for (ptr = class_head; ptr != NULL; ) {
	tmp = ptr;
	ptr = ptr->next;
	pse_free(tmp);
    }
    
    class_head = class_tail = NULL;
}

Class *clsRegisterClass(const char *name, Class *parent, int flags, 
			const FieldDef *def_table, 
			int (*validate_fun)(Record *),
			void (*add_fun)(Record *), 
			void (*reload_fun)(Record *), 
			int (*remove_fun)(Record *))
{
    int i;
    int max_index = -1;

    Class *newclass;

    newclass = (Class *) pse_malloc(sizeof(Class));
    
    strncpy(newclass->name, name, NAME_LEN);
    newclass->name[NAME_LEN - 1] = '\0';
    newclass->flags = flags;
    newclass->validate = validate_fun;
    newclass->on_add = add_fun;
    newclass->on_reload = reload_fun;
    newclass->on_remove = remove_fun;
    newclass->parent = parent;
    newclass->next = NULL;
    newclass->record_head = NULL;
    newclass->record_tail = NULL;

    btreeInsert(newclass->name, (void *)newclass, &class_names);

    if (class_head == NULL) 
	class_head = class_tail = newclass;
    else
	class_tail = class_tail->next = newclass;

    /*
     * Create definition table 
     */

    /* Find the largest index number and allocate an index of that size + 1. */

    for (i = 0; def_table[i].index >= 0; i++) {
	if (def_table[i].index > max_index)
	    max_index = def_table[i].index;
    }

    /* Allocate new definition table and zero it. */
    newclass->nfields = max_index + 1;
    newclass->defs = (const FieldDef **) pse_calloc(newclass->nfields, sizeof(FieldDef *));
    memset(newclass->defs, 0, sizeof(FieldDef *) * newclass->nfields); 
    
    for (i = 0; def_table[i].index >= 0; i++) 
	newclass->defs[def_table[i].index] = &def_table[i];

    return newclass;
}

int clsAddRecord(dbref player, dbref record)
{    
    Class *class;
    Record *recptr;
    
    class = clsClassOf(record);
    
    if (class == NULL) {
	Notify(player, "The object has not been set up properly.");
	log_space("AddObject Failure: #%d: CF_CLASS not found.", record);
	return 1;
    }

    if ( (class->flags & CLS_PUBLIC) == 0) {
	Notify(player, "Objects of this class cannot be added with add_object.");
	log_space("AddObject Failure: #%d: Class is private.", record);
	return 1;
    }

    if (clsRecordExists(class, record)) {
        Notify(player, "The object has already been added.");
	log_space("AddRecord Failure: #%d: Object already added.", record);
	return 1;
    } 

    if ( (recptr = clsLoadAndValidate(class, record)) == NULL) {
	Notify(player, "The object has not been set up properly.");
	return 1;
    }

    clsAddRecordDirect(recptr);

    return 0;
}

Class *clsClassOf(dbref record)
{
    char *buff;
    Class *retval;
    
    buff = pse_malloc(MAX_ATTRIBUTE_LEN);
    
    /* Try to find the default class. */
    getAttrByNameLen(buff, record, "CF_CLASS", MAX_ATTRIBUTE_LEN);
    
    if (buff[0] == '\0') {
	log_space("clsClassOf Failure: #%d: No CF_CLASS attribute set.",
		  record);
	retval = NULL;
    } else
	retval = (Class *)btreeSearch(buff, class_names);
    
    pse_free(buff);
    
    return retval;
}

int clsRecordExists(Class *class, dbref record)
{
    Record *ptr;

    /* Is record there already? (whoops!) */
    for (ptr = class->record_head; ptr != NULL; ptr = ptr->next) {
	if (ptr->db_obj == record)
	    return 1;
    }
    
    return 0;

}

Record *clsLoadAndValidate(Class *class, dbref record)
{
    Record *newrecptr;
 
    /* Load the new record. */
    if ( (newrecptr = load_record(class, record)) == NULL)
	return(NULL);
	
    /* Call validate, if okay, proceed. */
    if (class->validate(newrecptr) != 0) {
	free_record(newrecptr);
	return(NULL);
    }

    return newrecptr;
}

void clsAddRecordDirect(Record *recptr)
{
    Class *class = recptr->class;

    /* Add record to list. */
    if (class->record_head == NULL) 
	class->record_head = class->record_tail = recptr;
    else {
	recptr->prev = class->record_tail;
	class->record_tail = class->record_tail->next = recptr;
    }

    /* Call on_add. */
    if (class->on_add != NULL)
	class->on_add(recptr);

}

void clsQueue(Record *parent, Record *child)
{
    Record *ptr;

    if (parent->queue == NULL)
	parent->queue = child;
    else {
	for (ptr=parent->queue; ptr->next != NULL; ptr=ptr->next)
	    ;

	ptr->next = child;
    }

    child->next = NULL;
}

Record *clsRetrieve(Record *parent)
{
    Record *ptr;

    if (parent->queue == NULL)
	return NULL;
    else {
	ptr = parent->queue;
	parent->queue = parent->queue->next;
	ptr->next = NULL;
	return ptr;
    }
}

#ifdef NOT_IMPLEMENTED_YET
void clsReloadRecord(dbref player, dbref record)
{
    Class *class;
    Record *ptr, *oldrecptr = NULL;
    char *buff;
    
    class = clsClassOf(record);

    if (class == NULL) {
	Notify(player, bad_record);
	log_space("ReloadRecord Failure: #%d: CF_CLASS not found.", record);
	return;
    }

    /* Find the existing record. */
    for (ptr = class->record_head; ptr != NULL; ptr = ptr->next) {
	if (ptr->db_obj == record) {
	    oldrecptr = ptr;
	    break;
	}
    }

    if (oldrecptr == NULL) {
	Notify(player, "The object has not been added yet.");
	log_space("ReloadRecord Failure: #%d: Object not found.", record);
	return;
    }

    clsReloadRecordDirect(player, class, oldrecptr);

}
    
void clsReloadRecordDirect(dbref player, Class *class, Record *oldrecptr)
{
    Record *newrecptr;

    /* Load the new record. */
    if ( (newrecptr = load_record(class, oldrecptr->db_obj)) == NULL) {
	Notify(player, bad_record);
	return;
    }
	
    /* Call validate, if okay, proceed. */
    if (class->validate(newrecptr) != 0) {
	Notify(player, bad_record);
	free_record(newrecptr);
	return;
    }

    /* Carefully paste the new record in over top the old one. */
    if (oldrecptr->prev != NULL)
	oldrecptr->prev->next = newrecptr;
    else
	class->record_head = newrecptr;
    if (oldrecptr->next != NULL)
	oldrecptr->next->prev = newrecptr;
    else
	class->record_tail = newrecptr;
    newrecptr->prev = oldrecptr->prev;
    newrecptr->next = oldrecptr->next;
    free_record(oldrecptr);
    
    /* Call on_reload */
    if (class->on_reload != NULL)
	class->on_reload(newrecptr);
}
#endif

Record *clsFindRecord(Class *class, dbref record)
{
    Record *ptr;

    /* Find the existing record. */
    for (ptr = class->record_head; ptr != NULL; ptr = ptr->next) {
	if (ptr->db_obj == record)
	    return ptr;
    }

    return NULL;

}

int clsRemoveRecord(Record* recptr)
{
    int rval;

    /* Call on_remove. */
    if (recptr->class->on_remove != NULL) {

	/* Returns 0 if record is to be removed, -1 if record should not be 
	 * removed (no error), 1 if object should not be removed (error).
	 */
	rval = recptr->class->on_remove(recptr);
	if (rval == 1)
	    return 1;
	else if (rval == -1)
	    return 0;
    }

    /* Remove from list. */
    if (recptr->prev != NULL)
	recptr->prev->next = recptr->next;
    else
	recptr->class->record_head = recptr->next;
    if (recptr->next != NULL)
	recptr->next->prev = recptr->prev;
    else
	recptr->class->record_tail = recptr->prev;

    clsWriteRecord(recptr);
    free(recptr);
    return 0;
}

void clsWriteRecord(Record *recptr)
{
    int i;
    char *wbuf;
    
    wbuf = pse_malloc(MAX_ATTRIBUTE_LEN);

    /* Write back fields flagged as such. */
    for (i = 0; i < recptr->class->nfields; i++) {
	if ((recptr->fields[i].flags & FF_WRITE_BACK) > 0) {
	    switch (FD_TypeOf(recptr->class->defs[i]->flags)) {
	    case FD_TYPE_INT:
		snprintf(wbuf, MAX_ATTRIBUTE_LEN, "%d", F_INT(recptr, i));
		setAttrByName(recptr->db_obj, recptr->class->defs[i]->attr, wbuf);
		break;
	    case FD_TYPE_FLOAT:
		snprintf(wbuf, MAX_ATTRIBUTE_LEN, "%f", F_FLOAT(recptr, i));
		setAttrByName(recptr->db_obj, recptr->class->defs[i]->attr, wbuf);
		break;
	    case FD_TYPE_STRING:
		setAttrByName(recptr->db_obj, recptr->class->defs[i]->attr, F_PTR(recptr, i));
		break;
	    }
	}
    }

    pse_free(wbuf);
    return;
}

void clsSetString(Record *recptr, int index, const char *value)
{
    if ((recptr->fields[index].flags & FF_FREE_ME) > 0)
	pse_free(recptr->fields[index].u.pval);

    recptr->fields[index].u.pval = pse_strdup(value);
    recptr->fields[index].flags |= FF_FREE_ME | FF_EXISTS;
}

Record *load_record(Class *class, dbref record)
{
    int i;
    char *buff;
    Record *newrecptr;    
    
    /* Create a new record */
    newrecptr = (Record *) pse_malloc(sizeof(Record));
    newrecptr->class = class;
    newrecptr->db_obj = record;
    newrecptr->fields = (FIELD *) pse_calloc(class->nfields, sizeof(FIELD));
    newrecptr->queue = newrecptr->next = newrecptr->prev = NULL;
    
    /* Load attrs */
    buff = pse_malloc(MAX_ATTRIBUTE_LEN);
    getObjectName(buff, record, MAX_ATTRIBUTE_LEN);

    /* Remove any ansi codes in-situ */
    pse_strip_ansi(buff);

    strncpy(newrecptr->name, buff, NAME_LEN-1);
    newrecptr->name[NAME_LEN-1] = '\0';
    
    pse_free(buff);
    
    for (i = 0; i < class->nfields; i++) {
	if (clsReloadField(newrecptr, i) > 0) {
	    free_record(newrecptr);
	    return(NULL);
	}
    }    
    
    return newrecptr;
}

int clsReloadField(Record *recptr, int index)
{
    const char* attr;
    char* str;

    if (recptr->class->defs[index] == NULL)
	return 0;

    attr = recptr->class->defs[index]->attr;
    
    str = pse_malloc(MAX_ATTRIBUTE_LEN);
    
    getAttrByNameLen(str, recptr->db_obj, attr, MAX_ATTRIBUTE_LEN);
    if (str[0] == '\0') {
	if ((recptr->class->defs[index]->flags & FD_REQUIRED) > 0) {
	    log_space("LoadRecord Failure: #%d: Required attribute '%s' not present.", recptr->db_obj, attr);
	    pse_free(str);
	    return(1);
	}
    }
    else {
	switch (FD_TypeOf(recptr->class->defs[index]->flags)) {
	case FD_TYPE_INT:
	    /* Make sure integer is really an integer. */
	    if (clsValidateInt(str)) {
		log_space("LoadRecord Failure: #%d: Value of attribute '%s' must be an integer.", recptr->db_obj, attr);
		pse_free(str);
		return(1);
	    }
	    F_INT(recptr, index) = atoi(str);
	    break;
	case FD_TYPE_FLOAT:
	    /* Make sure float is really a float. */
	    if (clsValidateFloat(str)) {
		log_space("LoadRecord Failure: #%d: Value of attribute '%s' must be a float.", recptr->db_obj, attr);
		pse_free(str);
		return(1);
	    }
	    F_FLOAT(recptr, index) = (float) atof(str);
	    break;
	case FD_TYPE_STRING:
	    clsSetString(recptr, index, str);
	    pse_free(str);
	    break;
	default:
	    log_space("LoadRecord Failure: %d: Class '%s', field '%s' has invalid type value.", recptr->db_obj, recptr->class->name, attr);
	    pse_free(str);
	    return(1);
	}
	recptr->fields[index].flags |= FF_EXISTS;
    }
    
    return(0);

}

void free_record(Record *recptr)
{
    int i;

    for (i = 0; i < recptr->class->nfields; i++) {
	if ((recptr->fields[i].flags & FF_FREE_ME) > 0)
	    pse_free(recptr->fields[i].u.pval);
    }

    pse_free(recptr->fields);
    pse_free(recptr);

}

int clsValidateDbref(const char *dbrefstr)
{
    int i;
    dbref num;

    if (dbrefstr[0] == '\0')
	return 1;

    if (dbrefstr[0] != '#')
	return 1;

    for (i = 1; i < strlen(dbrefstr); i++) {
	if (!isdigit(dbrefstr[i]))
	    return 1;
    }

    num = atoi(&dbrefstr[1]);

    if (!ValidObject(num))
	return 1;

    return 0;
}

/* TODO:  In the next two functions it might be wise to make sure the
 * numbers actually fit within an int or a float, respectively.
 */
int clsValidateInt(const char *str)
{
    char *endptr;

    strtol(str, &endptr, 10);
    return (*endptr == '\0') ? 0 : 1;
}

int clsValidateFloat(const char *str)
{
    char *endptr;

    strtod(str, &endptr);
    return (*endptr == '\0') ? 0 : 1;
}

int clsValidateList(const char *list, int (*validate_item)(const char *item))
{
    int i, lasti = 0;
    char *wbuf;

    wbuf = pse_malloc(SMALL_BUF_SIZE);

    for (i = 0; i <= strlen(list); i++) {
	if (list[i] == ' ' || list[i] == '\0') {
	    if (i - lasti > SMALL_BUF_SIZE - 1) {
		pse_free(wbuf);
		return 1;
	    }
	    strncpy(wbuf, &(list[lasti]), i - lasti);
	    wbuf[i - lasti] = '\0';
	    if (validate_item(wbuf)) {
		pse_free(wbuf);
		return 1;
	    }
	    lasti = i + 1;
	}
    }

    pse_free(wbuf);
    return 0;
}
