/* v0.9
 *
 * damage.c:  Ship damage code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "damage.h"
#include "events.h"
#include "nav.h"
#include "shields.h"
#include "eng.h"
#include "sensors.h"
#include "smisc.h"
#include "tactical.h"

void repair_system(TAG *, int);
int hull_damage(TAG *, int, int);
void system_damage(TAG *, int);
void blast_check(TAG * , XYZ, int);

void damInitSystemDamage(TAG *object)
{
    int i;
    SHIP *ship = object->shipdata;
    
    /* engine stats */
    ship->current_reactor_output_max = ship->reactor_output_max *
	(1 - ((float)(ship->damage[SYSTEM_ENGINE].status - '0') * 0.1));
    ship->reactor_output = (int)(0.05 * ship->current_reactor_output_max);
    
    /* battery stuff */
    ship->battery_discharge = 0;
    ship->battery_level = ship->battery_capacity;
    
    switch (ship->damage[SYSTEM_BATTERIES].status) {
    case '9':
    case '7':
	ship->battery_status = BTTY_DAMAGED;
	ship->battery_level = 0;
	break;
    case '6':
	ship->battery_discharge_max *= 0.5;
	ship->battery_status = BTTY_OFFLINE;
	break;
    case '1':
	ship->battery_discharge_max *= 0.9;
	ship->battery_status = BTTY_OFFLINE;
	break;
    case '0':
	ship->battery_status = BTTY_OFFLINE;
	break;
    }
    
    /* sensors/scanners */
    switch (ship->damage[SYSTEM_SENSORS].status) {
    case '1':
	object->sensor_range *= 0.9;
	break;
    case '6':
	object->sensor_range *= 0.5;
	break;
    }
    
    switch (ship->damage[SYSTEM_SCANNERS].status) {
    case '1':
	ship->scanner_range *= 0.9;
	break;
    case '6':
	ship->scanner_range *= 0.5;
	break;
    }
    
    /* transporter */
    switch (ship->damage[SYSTEM_TRANSPORTER].status) {
    case '1':
	object->transporter_range *= 0.9;
	break;
    case '6':
	object->transporter_range *= 0.5;
	break;
    }
    
    /* tractor beam */
    switch (ship->damage[SYSTEM_TRACTOR].status) {
    case '1':
	ship->tractor_effect *= 0.9;
	break;
    case '6':
	ship->tractor_effect *= 0.5;
	break;
    case '9':
    case '7':
	ship->tractor_status = TRACTOR_DAMAGED;
	break;
    case 'x':
	if (ship->tractor_status != TRACTOR_NONE)
	    ship->damage[SYSTEM_TRACTOR].status = '0';
	break;
    }
    
    
    if (ship->damage[SYSTEM_CLOAK].status == 'x' && 
	ship->cloak_status != CLOAK_NONE)
	ship->damage[SYSTEM_CLOAK].status = '0';
    
    /* adjust the capabilities of any damaged shields */
    for (i = 0; i < NUM_SHIELDS; i++) {
	switch (ship->damage[SYSTEM_SHIELD_FORE + i].status) {
	case '9':
	case '7':
	    ship->shield_status[i] = SHLD_DAMAGED;
	    break;
	case '6':
	    ship->shield_max[i] *= 0.5;
	    ship->shield_max_charge[i] *=0.5;
	    break;
	case '1':
	    ship->shield_max[i] *= 0.9;
	    ship->shield_max_charge[i] *=0.9;
	    break;
	case '0':
	    break;
	}
    }
    
    /* check for gun damage */
    for (i = 0; i < ship->number_of_guns; i++) {
	
	switch (ship->damage[SYSTEM_GUN_0 + i].status) {
	case '9':
	case '7':
	    ship->gun[i].status = GUN_DAMAGED;
	    break;
	case '6':
	    ship->gun[i].power *= 0.5;
	    ship->gun[i].charge_per_turn *= 0.5;
	    ship->gun[i].range *= 0.5;
	    break;
	case '1':
	    ship->gun[i].power *= 0.9;
	    ship->gun[i].charge_per_turn *= 0.9;
	    ship->gun[i].range *= 0.9;
	    break;
	case 'x':
	    ship->damage[SYSTEM_GUN_0 + i].status = '0';
	    break;
	}
    }
    
    for (i = 0; i < ship->number_of_torps; i++) {
	
	/* check for damage */
	switch (ship->damage[SYSTEM_TORP_0 + i].status) {
	case '9':
	case '7':
	    if (ship->torp[i].status == TORP_LOADING)
		ship->free_torp_loaders++;
	    ship->torp[i].status = TORP_DAMAGED;
	    break;
	case '1':
	    ship->torp[i].range *= 0.9;
	    break;
	case 'x':
	    ship->damage[SYSTEM_TORP_0 + i].status = '0';
	    break;
	}
    }
    
    return;
}

void damWriteDamage(dbref data, SHIP *ship)
{
    char buff[SMALL_BUF_SIZE];
    int i;
    
    for (i = 0; i < NUM_SYSTEMS; i++)
	buff[i] = ship->damage[i].status;
    
    for (i = ship->shield_number; i < NUM_SHIELDS; i++)
	buff[i] = 'X';
    
    buff[NUM_SYSTEMS] = '\0';
    
    setAttrByName(data, STATUS_SYSTEM_DAMAGE, buff);
    
    sprintf(buff, "%d %d", ship->hull_integrity - 
	    ship->current_integrity, ship->permanent_hull_damage);
    
    setAttrByName(data, STATUS_HULL_DAMAGE, buff);
}

void damShipRepairs(int space)
{
     SHIP *ship;
     TAG *object;
     int i;

     for (object = space_info[space].list; object != NULL; object=object->next) {

	  if (Removed(object))
	       continue;

	  if (!Ship(object))
	       continue;

	  ship = object->shipdata;

	  for (i = 0; i < ship->damcon_teams; i++) {
	       /* for each team, deduct time of the job they're on */
	       if (ship->team[i] == -1) 
		    continue;

	       /* Do hull separately */
	       if (ship->team[i] == SYSTEM_HULL) {
		    ship->current_integrity++;

		    if (Disabled(ship) && 
			ship->current_integrity > 0)
			 ship->shipflags &= ~DISABLED;

		    if (ship->current_integrity + 
			ship->permanent_hull_damage >=
			ship->hull_integrity) 
			 repair_system(object, SYSTEM_HULL);

		    continue;
	       }

	       /* They're on a job.  Deduct time from it */
	       ship->damage[ship->team[i]].time_to_fix -= 6;

	       /* now check and see if the job is finished */
	       if (ship->damage[ship->team[i]].time_to_fix < 1 &&
		   ship->damage[ship->team[i]].status != '9')
		    repair_system(object, ship->team[i]);
	  }

     }

     return;

}

void repair_system(TAG *object, int system)
{
     char buff[SMALL_BUF_SIZE];
     SHIP *ship = object->shipdata;
     int i;

     /* switch on the system and effect the repair */
     switch (system) {
     case SYSTEM_GUN_0:
     case SYSTEM_GUN_1:
     case SYSTEM_GUN_2:
     case SYSTEM_GUN_3:
     case SYSTEM_GUN_4:
     case SYSTEM_GUN_5:
	  ship->gun[system - SYSTEM_GUN_0].status = GUN_OFFLINE;
	  ship->gun[system - SYSTEM_GUN_0].power = 
	       atoi(getAttrByName(object->data_object, CONFIG_GUN_POWER)) * 
	       0.9;
	  ship->gun[system - SYSTEM_GUN_0].range = 
	       atoi(getAttrByName(object->data_object, CONFIG_GUN_RANGE)) * 
	       0.9;
	  ship->gun[system - SYSTEM_GUN_0].charge = 0;
	  ship->gun[system - SYSTEM_GUN_0].charge_per_turn = 
	       atoi(getAttrByName(object->data_object, 
				  CONFIG_GUN_CHARGE_RATE)) * 0.9;

	  sprintf(buff, "[%s repaired]", system_names[system]);

	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_TORP_0:
     case SYSTEM_TORP_1:
     case SYSTEM_TORP_2:
     case SYSTEM_TORP_3:
     case SYSTEM_TORP_4:
     case SYSTEM_TORP_5:
	  ship->torp[system - SYSTEM_TORP_0].status = TORP_EMPTY;
	  ship->torp[system - SYSTEM_TORP_0].range = 
	       atoi(getAttrByName(object->data_object, CONFIG_TORP_RANGE)) * 
	       0.9;
	  ship->torp[system - SYSTEM_TORP_0].charge = 0;
	  ship->torp[system - SYSTEM_TORP_0].turns_charged = 0;
	  ship->torp[system - SYSTEM_TORP_0].base_accuracy = 
	       atof(getAttrByName(object->data_object, CONFIG_TORP_ACCURACY))
	       * 0.9;

	  sprintf(buff, "[%s repaired]", system_names[system]);

	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_SCANNERS:
	  sprintf(buff, "[%s repaired]", system_names[system]);

	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_SENSORS:
	  object->sensor_range = atof(getAttrByName(object->data_object, 
				   CONFIG_SENSOR_RANGE)) * 0.9;
	  object->tagflags |= CAN_SENSE;
	  sprintf(buff, "[%s repaired]", system_names[system]);

	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_TRACTOR:
	  ship->tractor_effect = atof(getAttrByName(object->data_object,
						    CONFIG_TRACTOR_EFFECT)) 
	       * 0.9;

	  if (ship->tractor_status == TRACTOR_ON || ship->tractor_status == 
	      TRACTOR_DRAW)
	       ship->tractor_power = (int)((float) ship->talloc_tractor * 
					   ship->tractor_effect / 
					   ship->tractor_target->size);

	  sprintf(buff, "[%s repaired]", system_names[system]);

	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_TRANSPORTER:
	  object->transporter_range=atof(getAttrByName(object->data_object,
						       CONFIG_XPORT_RANGE)) 
	       * 0.9;

	  sprintf(buff, "[%s repaired]", system_names[system]);

	  MFNotify(ship, -1, CONS_TRANS | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_SHIELD_FORE:
     case SYSTEM_SHIELD_AFT:
     case SYSTEM_SHIELD_PORT:
     case SYSTEM_SHIELD_STARBOARD:
     case SYSTEM_SHIELD_DORSAL:
     case SYSTEM_SHIELD_VENTRAL:
	  ship->shield_max[system - SYSTEM_SHIELD_FORE] = 
	       atoi(getAttrByName(object->data_object, CONFIG_SHIELD_POWER)) 
	       * 0.9;
	  ship->shield_max_charge[system - SYSTEM_SHIELD_FORE] = 
	       atoi(getAttrByName(object->data_object, 
				  CONFIG_SHIELD_CHARGE_RATE)) * 0.9;
	  ship->shield_level[system - SYSTEM_SHIELD_FORE] = 0;
	  ship->shield_status[system - SYSTEM_SHIELD_FORE] = SHLD_READY;
	  ship->shield_action[system - SYSTEM_SHIELD_FORE] = SHLD_STABLE;

	  sprintf(buff, "[%s shield repaired]", shdFullName(object,system));

	  MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_CLOAK:
	  ship->cloak_status = CLOAK_OFF;

	  sprintf(buff, "[%s repaired]", system_names[system]);

	  MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_ENGINE:
	  ship->damage[SYSTEM_ENGINE].status--;
	  /* recalculate max_engine_output */
	  ship->current_reactor_output_max = ship->reactor_output_max *
	       (1-((float)(ship->damage[SYSTEM_ENGINE].status - '0')*0.1));

	  sprintf(buff, "[Engine partially repaired]");

	  MFNotify(ship, -1, CONS_ENG | CONS_DAM, CONS_ACTIVE, buff);
	  break;

     case SYSTEM_HULL:
	  /* This one is a little different.  There's no status change.
	   * This function is simply called when the patch is as good
	   * as it can get.  This notifies and removes teams from the job.
	   */

	  MFNotify(ship, -1, CONS_DAM, CONS_ACTIVE, 
		   "[Hull patches completed]");

	  break;
     }

     /* remove all teams from the job */
     for (i = 0; i < ship->damcon_teams; i++) {
	  if (ship->team[i] == system) {
	       ship->team[i] = -1;
	       MFNotify(ship, -1, CONS_DAM, CONS_ACTIVE, 
			"[Team %d now free]", i);
	  }
     }

     /* reset the damage register */
     ship->damage[system].time_to_fix = 0;

     if (system != SYSTEM_HULL && system != SYSTEM_ENGINE)
	  ship->damage[system].status = '1';
  
     ship->damage[system].teams = 0;

     return;
}


void damBattleDamage(XYZ source, TAG *target, int damage, int weapon)
{
     int shield_hit;
     SHIP *ship;
     int damage_orig = damage;

     shield_hit = calcFacingShield(source, target);

     if (Ship(target)) {
	  ship = target->shipdata;

	  /* if the target is set invincible, set the damage to zero */
	  if (Invincible(ship))
	       damage = 0;

	  /* if the shield is up and charged, do it this way */
	  if (ship->shield_status[shield_hit] == SHLD_UP &&
	      ship->shield_level[shield_hit] > 0 &&
	      !Naked(ship)) {

	       if (weapon == WPN_TORP) {
		    /*
		     * If the torp has enough energy to knock down the 
		     * shield only 10% of the excess gets through
		     */
		    if (damage > ship->shield_level[shield_hit])
			 damage -= (damage - 
				    ship->shield_level[shield_hit]) * 0.9;
	       }
	       else if (weapon == WPN_GUN) {
		    /* guns aren't too hot against shields.  
		     * Knock 10% off.
		     */
		    damage *= 0.9;
		    if (damage > ship->shield_level[shield_hit])
			 damage -= (damage - 
				    ship->shield_level[shield_hit]) * 0.1;
	       }
	       else if (weapon == EXPLOSION) {
		    /* explosions are absolutely horrible against shields.
		     * knock 90% off.
		     */
		    damage *= 0.1;
		    if (damage > ship->shield_level[shield_hit])
			 damage -= (damage - 
				    ship->shield_level[shield_hit]) * 0.9;
	       }

	       if (damage < ship->shield_level[shield_hit]) {

		    MFNotify(ship, -1, CONS_SHD, CONS_ACTIVE,
			     "[%s shield reduced %d]", shdFullName(target,
								   shield_hit)
			     , damage);

		    ship->shield_level[shield_hit] -= damage;
		    /* let 1% through if the shield isn't overloaded,
		     * 2% if it is.
		     */
		    if (ship->shield_level[shield_hit] > 
			ship->shield_max[shield_hit])
			 hull_damage(target, damage / 50, INFO_TERSE);
		    else
			 hull_damage(target, damage / 100, INFO_TERSE);
	       }
	       else {
		    MFNotify(ship, -1, CONS_SHD, CONS_ACTIVE,
			     "[%s shield failed]", shdFullName(target,
							       shield_hit));

		    damage -= ship->shield_level[shield_hit];
		    ship->shield_level[shield_hit] = 0;
		    if (damage)
			 hull_damage(target, damage, INFO_VERBOSE);
	       }
	  }
	  else { /* shield wasn't up or something */
	       if (weapon == WPN_TORP)
		    damage *= 0.9;
	       MFNotify(ship, -1, CONS_SHD, CONS_ACTIVE,
			"[Direct hit to %s]", shdFullName(target,shield_hit));

	       hull_damage(target, damage, INFO_VERBOSE);
	  }
     }

     /* Only non-destroyed objects take this event */
     if (!Removed(target))
	 evTrigger(target, EVENT_TAKE_DAMAGE, "ii", damage_orig, weapon);

     return;
}

int hull_damage(TAG *object, int damage, int control)
{
     char buff[SMALL_BUF_SIZE];
     SHIP *ship = object->shipdata;
     int space, output;
     XYZ pos;
     dbref man;

     /* this is simple for now.  Just subtract the damage from the current
      * total, notify everyone, and kill the ship if it's negative
      */

     /* return if damage is zero.  This will happen often due to the damage
      * bleeding through full shields.
      */

     if (damage == 0)
	  return 0;

     ship->current_integrity -= damage;

     /* update the permanent damage register */
     ship->permanent_hull_damage += (damage / 10);

     /* if we're destroyed... */
     /* note that you aren't actually dead until damage hits -33 1/3% */
     if (ship->current_integrity <= (ship->hull_integrity / -3)) {

	  /* If in real space, log it */
	  if (space_info[object->space].flags & SPACE_LOGGED) {
	       log_space("%s(#%d) destroyed. Manned consoles:", object->name, 
			 object->data_object);

	       for (output = 0; output < ship->num_consoles; output ++) {
		    man = dbrefUser(ship->consoles[output].dbref);
		    log_space("Console %d(#%d): %s(#%d)",
			      output, ship->consoles[output].dbref,
			      getObjectName(man), man);
	       }
	  }

	  evTrigger(object, EVENT_DESTROYED, "");

	  space = object->space;
	  pos.x = object->pos.x;
	  pos.y = object->pos.y;
	  pos.z = object->pos.z;
	  output = ship->reactor_output;

	  /* Remove the ship from the firing table. */
	  tacRemoveFireTableEntry(object);

	  objRemoveObject(object);
	  damExplosionDamage(space, pos, output);
	  return 1;
     }

     if (control == INFO_VERBOSE) {

	  sprintf(buff, "Hull damaged %d -- integrity now at %d%%",
		  damage, (100 * ship->current_integrity / 
			   ship->hull_integrity));

	  NotifyLocation(dbrefBridge(ship), buff);
	  if (dbrefBridge(ship) != dbrefEngine(ship))
	      NotifyLocation(dbrefEngine(ship), buff);
     }

     /* now break some ship systems */
     system_damage(object, damage);
							     
     /* now check and see if the target is disabled */
     if (ship->current_integrity <= 0) {

	 /* Only emit and log if we were not disabled to start with. */
	 if (!Disabled(ship)) {
	     sprintf(buff, "Ship disabled.");
	  
	     NotifyLocation(dbrefBridge(ship), buff);
	     if (dbrefBridge(ship) != dbrefEngine(ship))
		 NotifyLocation(dbrefEngine(ship), buff);
	     
	     log_space("%s (#%d) has been disabled.", 
		       object->name, object->data_object);
	 }
	 
	 engAllocate(object, PSE_NOTHING, 0, 0, 0, 0);
	 
	 if (!Disabled(ship)) {
	     ship->shipflags |= DISABLED;
	     evTrigger(object, EVENT_DISABLED, "i", 1);
	 }
	 else
	     evTrigger(object, EVENT_DISABLED, "i", 0);
     }
     
     return 0;
}

void system_damage(TAG *object, int damage)
{
     SHIP *ship = object->shipdata;
     int before, after;   /* used in damage calculation */
     float ratio;         /* fraction lost in the attack */
     float chance;        /* chance of losing a given system */
     int i;

     after = ship->current_integrity;
     before = after + damage;

     ratio = (float)damage / (float)before;

     chance = fabs(ratio * 0.3);

     /* now start a big long list of if's */
     /* shields */
     for (i = 0; i < NUM_SHIELDS; i++)
	  if (FRAND < chance)
	       damDamageShield(object, i, 1);

     if (FRAND < chance)
	  damDamageCloak(object, 1);

     if (FRAND < chance)
	  damDamageBattery(object, 1);

     for (i = 0; i < ship->number_of_guns; i++)
	  if (FRAND < chance)
	       damDamageGun(object, i, 1);

     for (i = 0; i < ship->number_of_torps; i++)
	  if (FRAND < chance)
	       damDamageTorp(object, i, 1);

     if (FRAND < chance / 2.0)
	  damDamageScanner(object, 1);

     if (FRAND < chance / 3.0)
	  damDamageSensor(object, 1);

     if (FRAND < chance / 2.0)
	  damDamageTransporter(object, 1);

     if (FRAND < chance / 2.0)
	  damDamageTractor(object, 1);

     if (FRAND < chance / 3.0)
	  damDamageEngine(object, (int)((FRAND * 3) + 2));

     return;
}

void damRepair(TAG *object, dbref player, int team, char *system)
{
     SHIP *ship = object->shipdata;
     int sys;

     sys = objSystemNameToInt(object, system);

     if (sys < 0) {
	  switch(sys) {
	  case -2:
	       Notify(player, "Invalid system.");
	       break;
		    
	  case -1:
	       Notify(player, "Ambiguous system.");
	       break;
		    
	  default:
	       Notify(player, "Unknown return from objSystemNameToInt in "
		      "damRepair.");
	       break;
	  }
		
	  return;
     }

     /* Okay, 'sys' is now the index of the system we want to fix.
      * First check to see if the team is already working, and if they are,
      * remove them from that job.  Then put them on the new one.
      */

     /* make sure the new job actually needs them */
     if (sys == SYSTEM_HULL) {
	  if (ship->current_integrity >= ship->hull_integrity - 
	      ship->permanent_hull_damage) {
	       Notify(player, "No futher hull repairs are possible.");
	       return;
	  }
     }
     else {
	  if (ship->damage[sys].status == '1' || 
	      ship->damage[sys].status == '0') {
	       FNotify(player, "%s is undamaged.", objSystemIntToName(object, 
								      sys));
	       return;
	  }
     }

     /* don't let them fix a destroyed system */
     if (ship->damage[sys].status == '9') {
	  Notify(player, "You can't repair a destroyed system.");
	  return;
     }

     /* make sure there's room on the new job to accomodate them */
     if (ship->damage[sys].teams == ship->damage[sys].maxteams) {
	  FNotify(player, "The maximum possible number of teams is already "
		  "working on %s.", objSystemIntToName(object, sys));
	  return;
     }

     if (ship->team[team] != -1) {
	  /* The team is on another job.  Remove them from it. */
	  ship->damage[ship->team[team]].teams--;
     }

     /* now put then on the new job */
     ship->damage[sys].teams++;
     ship->team[team]=sys;

     /* and notify the player */
     MFNotify(ship, player, CONS_DAM, CONS_ACTIVE,
	      "[Team %d now working on %s]", team,
	      objSystemIntToName(object, sys));

     return;
}

void damSystemStatus(TAG *object, dbref player)
{
     SHIP *ship = object->shipdata;
     int i;
     int count = 0;

     Notify(player, "Damage control system status:");
     Notify(player, "System            Condition\n"
	            "------            ---------");
     
     for (i = 0; i < ship->shield_number; i++) {
	  if (ship->damage[i].status > '0')
	       count++;

	  switch (ship->damage[i].status) {
	  case '0':   /* normal working order */
	       break;
	  case '1':   /* field repaired */
	       FNotify(player, "%-16s  field repair", shdFullName(object,i));
	       break;

	  case '2':   /* light damage */
	       FNotify(player, "%-16s  light damage", shdFullName(object,i));
	       break;

	  case '3':   /* moderate damage */
	  case '4':
	       FNotify(player, "%-16s  moderate damage",
		       shdFullName(object,i));
	       break;

	  case '5':   /* heavy damage */
	  case '6':
	       FNotify(player, "%-16s  heavy damage", shdFullName(object,i));
	       break;

	  case '7':   /* inoperable */
	  case '8':   
	       FNotify(player, "%-16s  inoperative", shdFullName(object,i));
	       break;

	  case '9':   /* destroyed */
	       FNotify(player, "%-16s  destroyed", shdFullName(object,i));
	       break;
	  }
     }

     for (i = NUM_SHIELDS; i < NUM_SYSTEMS; i++) {

	  /* hull is a special case.  Do it differenty. */
	  if (i == SYSTEM_HULL && ship->current_integrity < 
	      ship->hull_integrity) {
	       if (ship->hull_integrity - ship->permanent_hull_damage >
		   ship->current_integrity) {
				/* there's damage yet that we can repair */
		    FNotify(player, "%-16s  %d%% -- patchable",
			    system_names[i], 
			    (100 * ship->current_integrity / 
			     ship->hull_integrity));
	       }
	       else 
				/* no damage that we can repair */
		    FNotify(player, "%-16s  %d%% -- fully patched",
			    system_names[i], 
			    (100 * ship->current_integrity / 
			     ship-> hull_integrity));
	       continue;
	  }

	  if (ship->damage[i].status > '0' && 
	      ship->damage[i].status != 'x' &&
	      ship->damage[i].status != 'X')
	       count++;

	  switch (ship->damage[i].status) {
	  case '0':   /* normal working order */
	       break;
	  case '1':   /* field repaired */
	       FNotify(player, "%-16s  field repair", system_names[i]);
	       break;
	  case '2':   /* light damage */
	       FNotify(player, "%-16s  light damage", system_names[i]);
	       break;

	  case '3':   /* moderate damage */
	  case '4':
	       FNotify(player, "%-16s  moderate damage", 
		       system_names[i]);
	       break;

	  case '5':   /* heavy damage */
	  case '6':
	       FNotify(player, "%-16s  heavy damage", system_names[i]);
	       break;

	  case '7':   /* inoperable */
	  case '8':   
	       FNotify(player, "%-16s  inoperative", system_names[i]);
	       break;

	  case '9':   /* destroyed */
	       FNotify(player, "%-16s  destroyed", system_names[i]);
	       break;
	  }
     }

     if (count==0)
	  Notify(player, "All systems in normal working order.");

     Notify(player, " ");

     return;
}

void damTeamStatus(TAG *object, dbref player)
{
     SHIP *ship = object->shipdata;
     int i;

     if (ship->damcon_teams == 0)
	  return;

     Notify(player, "Damage control team status:");
     Notify(player, "Team      Job\n----     -----");

     for (i = 0; i < ship->damcon_teams; i++) {
	  if (ship->team[i] == -1)
	       FNotify(player, "Team %d:  Idle", i);
	  else
	       if (ship->team[i] < NUM_SHIELDS)
		    FNotify(player, "Team %d:  %s", i, 
			    shdFullName(object,ship->team[i]));
	       else
		    FNotify(player, "Team %d:  %s", i, 
			    system_names[ship->team[i]]);
     }

     Notify(player, " ");
     return;
}	

void damExplosionDamage(int space, XYZ point, int damage)
{
    TAG *ptr;
    int r, d;
    SPH bearing;
    
    xyz_to_sph(point, &bearing);
    
    if (space_info[space].flags & SPACE_LOGGED)
	log_space("A %d point explosion occured at %3.1f%+3.1f %d.",
		  damage, bearing.bearing, bearing.elevation, bearing.range);
    
    evExplosion(space, point, damage);
    
    r = HubRange(point);
    
    for (ptr=space_info[space].dist; ptr != NULL; ptr=ptr->range_next) {
	
	if (Removed(ptr))
	    continue;
	
	d = ptr->range - r;
	
	if (d >= damage)
	    break;
	
	if (-d >= damage)
	    continue;		
	
	blast_check(ptr, point, damage);
    }
    
    return;
}

void blast_check(TAG *target, XYZ point, int damage)
{
    int moddmg;
    
    if (!Attackable(target))
	return;
    
    moddmg = damage - distance(&target->pos, &point);
    
    if (moddmg > 0) {
	log_space("%s [#%d] takes %d points of explosion damage.",
		  target->name, target->data_object, moddmg);
	damBattleDamage(point, target, moddmg, EXPLOSION);
    }
    
    return;
}

void damSystemDamage(TAG *object, int system, int points)
{
     switch(system) {
     case SYSTEM_SHIELD_FORE:
     case SYSTEM_SHIELD_AFT:
     case SYSTEM_SHIELD_PORT:
     case SYSTEM_SHIELD_STARBOARD:
     case SYSTEM_SHIELD_DORSAL:
     case SYSTEM_SHIELD_VENTRAL:
	  damDamageShield(object, system - SYSTEM_SHIELD_FORE, points);
	  return;

     case SYSTEM_GUN_0:
     case SYSTEM_GUN_1:
     case SYSTEM_GUN_2:
     case SYSTEM_GUN_3:
     case SYSTEM_GUN_4:
     case SYSTEM_GUN_5:
	  damDamageGun(object, system - SYSTEM_GUN_0, points);
	  return;

     case SYSTEM_TORP_0:
     case SYSTEM_TORP_1:
     case SYSTEM_TORP_2:
     case SYSTEM_TORP_3:
     case SYSTEM_TORP_4:
     case SYSTEM_TORP_5:
	  damDamageTorp(object, system - SYSTEM_TORP_0, points);
	  return;

     case SYSTEM_BATTERIES:
	  damDamageBattery(object, points);
	  return;

     case SYSTEM_CLOAK:
	  damDamageCloak(object, points);
	  return;

     case SYSTEM_ENGINE:
	  damDamageEngine(object, points);
	  return;

     case SYSTEM_SENSORS:
	  damDamageSensor(object, points);
	  return;

     case SYSTEM_SCANNERS:
	  damDamageScanner(object, points);
	  return;

     case SYSTEM_TRACTOR:
	  damDamageTractor(object, points);
	  return;

     case SYSTEM_TRANSPORTER:
	  damDamageTransporter(object, points);
	  return;

     case SYSTEM_HULL:
	  hull_damage(object, points, INFO_TERSE);
	  return;
     }
}

int damDamageShield(TAG *object, int shield, int points)
{
     SHIP *ship = object->shipdata;

     /* Return now if the shield is already destroyed or not present */
     if ((ship->damage[SYSTEM_SHIELD_FORE + shield].status == '9') ||
	 (ship->damage[SYSTEM_SHIELD_FORE + shield].status == 'x') ||
	 (ship->damage[SYSTEM_SHIELD_FORE + shield].status == 'X'))
	  return 0;

     for (; points > 0; points--) {

	  switch (ship->damage[SYSTEM_SHIELD_FORE + shield].status) {
		
	  case '9': /* already gone.  Leave it alone */
	       points = 0;
	       break;

	  case '7': /* already damaged.  Waste it. */
	       ship->shield_level[shield] = 0;
	       ship->damage[SYSTEM_SHIELD_FORE + shield].status='9';
	       ship->shield_status[shield] = SHLD_DAMAGED;
	       break;

	  case '6': /* damaged once already.  send inoperable */
	  case '1':
	       ship->shield_level[shield] = 0;
	       ship->damage[SYSTEM_SHIELD_FORE + shield].status = '7';
	       ship->damage[SYSTEM_SHIELD_FORE + shield].time_to_fix = 600;
	       ship->shield_status[shield] = SHLD_DAMAGED;
	       break;

	  case '0':
	       ship->damage[SYSTEM_SHIELD_FORE + shield].status = '6';
	       ship->damage[SYSTEM_SHIELD_FORE + shield].time_to_fix = 300;
	       ship->shield_max[shield] *= 0.5;
	       ship->shield_max_charge[shield] *=0.5;

	       if (ship->shield_level[shield] >
		   (int)((float)ship->shield_max[shield] * 0.9))
		    ship->shield_level[shield] =
			 (int)((float)ship->shield_max[shield] * 0.9);
	       break;
	  }
     }

     /* Do the emits now. Only destroyed/inoperable/damaged are possible */

     switch (ship->damage[SYSTEM_SHIELD_FORE + shield].status) {

     case '9':	/* Shield was destroyed */
	  MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE,
		   "%s shield destroyed.", shdFullName(object, shield));

	  if (ship->cloak_status == CLOAK_ON) {
	       MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE,
			"Cloaking device failing due to shield loss.");
	       shdCloakOff(object, -1);
	  }

	  break;

     case '7':	/* Shield became inoperable */
	  MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE,
		   "%s shield inoperable.", shdFullName(object, shield));

	  if (ship->cloak_status == CLOAK_ON) {
	       MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE,
			"Cloaking device failing due to shield loss.");
	       shdCloakOff(object, -1);
	  }

	  break;

     case '6':	/* Shield was damaged */
	  MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE,
		   "%s shield damaged.", shdFullName(object, shield));
	  break;
     }

     return 0;
}

int damDamageCloak(TAG *object, int points)
{
     SHIP *ship = object->shipdata;

     /* Return now if the cloak is already destroyed or not present */
     if ((ship->damage[SYSTEM_CLOAK].status == '9') ||
	 (ship->damage[SYSTEM_CLOAK].status == 'x') ||
	 (ship->damage[SYSTEM_CLOAK].status == 'X'))
	  return 0;

     for (; points > 0; points --) {

	  switch (ship->damage[SYSTEM_CLOAK].status) {
	  case '9':	/* Cloak was destroyed */
	       points = 0;
	       break;

	  case '7':	/* already damaged. Waste it */
	       ship->damage[SYSTEM_CLOAK].status = '9';
	       break;

	  case '1':	/* send inoperable */
	  case '0':
	       ship->damage[SYSTEM_CLOAK].status = '7';
	       ship->damage[SYSTEM_CLOAK].time_to_fix = 600;
	       break;
	  }
     }

     /* Do the emits now. Only destroyed/inoperable are possible */

     switch (ship->damage[SYSTEM_CLOAK].status) {

     case '9':	/* Cloak was destroyed */
	  MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE,
		   "Cloak is destroyed.");

	  break;

     case '7':	/* Cloak was made inoperable */
	  MFNotify(ship, -1, CONS_SHD | CONS_DAM, CONS_ACTIVE,
		   "Cloak is inoperable.");

	  break;
     }

     if (ship->cloak_status == CLOAK_ON)
	  shdCloakOff(object, -1);

     return 0;
}

int damDamageBattery(TAG *object, int points)
{
     SHIP *ship = object->shipdata;

     switch (ship->damage[SYSTEM_BATTERIES].status) {
     case '9':   /* already dead */
	  break;
     case '7':   /* inoperable.  destroy */
	  ship->battery_level = 0;
	  ship->damage[SYSTEM_BATTERIES].status = '9';
	  ship->battery_status = BTTY_DAMAGED;

	  MFNotify(ship, -1, CONS_ENG | CONS_DAM, CONS_ACTIVE,
		   "Batteries destroyed.");

	  engAllocCheck(object);
	  break;

     case '6':   /* heavy dmg or field rep.  Send inoperable */
     case '1':
	  ship->damage[SYSTEM_BATTERIES].status = '7';
	  ship->damage[SYSTEM_BATTERIES].time_to_fix = 600;
	  ship->battery_status = BTTY_DAMAGED;

	  MFNotify(ship, -1, CONS_ENG | CONS_DAM, CONS_ACTIVE,
		   "Batteries seriously damaged.");

	  engAllocCheck(object);
	  break;
     case '0':   /* normal working condition.  Apply heavy dmg. */
	  ship->damage[SYSTEM_BATTERIES].status = '6';
	  ship->damage[SYSTEM_BATTERIES].time_to_fix = 300;
	  ship->battery_discharge_max /= 2;

	  MFNotify(ship, -1, CONS_ENG | CONS_DAM, CONS_ACTIVE,
		   "Batteries damaged.  Output reduced 50%.");

	  engAllocCheck(object);
	  break;
     }

     return 0;
}

int damDamageGun(TAG *object, int gun, int points)
{
     SHIP *ship = object->shipdata;

     switch (ship->damage[SYSTEM_GUN_0 + gun].status) {
     case '9':   /* already dead. */
	  break;
     case '7':   /* inoperable or hvy damage.  destroy it */
     case '6':
	  ship->damage[SYSTEM_GUN_0 + gun].status = '9';

	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "%s %d destroyed.", capstr(ship->gun_string), gun);

	  if (ship->gun[gun].status == GUN_ONLINE)
	       ship->num_guns_online--;
	  ship->gun[gun].status = GUN_DAMAGED;
	  ship->gun[gun].charge = 0;
	  break;

     case '1':   /* field repaired.  send inoperable */
	  ship->damage[SYSTEM_GUN_0 + gun].status = '7';
	  ship->damage[SYSTEM_GUN_0 + gun].time_to_fix = 600;

	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "%s %d damaged.  Inoperable.", capstr(ship->gun_string), 
		   gun);

	  if (ship->gun[gun].status == GUN_ONLINE)
	       ship->num_guns_online--;
	  ship->gun[gun].status = GUN_DAMAGED;
	  ship->gun[gun].charge = 0;
	  break;

     case '0':   /* normal working condition.  damage badly. */
	  ship->damage[SYSTEM_GUN_0 + gun].status = '6';
	  ship->damage[SYSTEM_GUN_0 + gun].time_to_fix = 300;
	  ship->gun[gun].range /= 2;
	  ship->gun[gun].power /= 2;
	  ship->gun[gun].charge_per_turn /= 2;

	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "%s %d heavily damaged.", capstr(ship->gun_string), gun);

	  break;
     }

     return 0;
}

int damDamageTorp(TAG *object, int torp, int points)
{
     SHIP *ship = object->shipdata;

     switch (ship->damage[SYSTEM_TORP_0 + torp].status) {
     case '9':   /* already dead */
	  break;
     case '7':   /* damaged.  Destroy */
	  ship->torp[torp].status = TORP_DAMAGED;
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "%s %d destroyed.", capstr(ship->torp_string), torp);
	  ship->torp[torp].charge = 0;
	  ship->damage[SYSTEM_TORP_0 + torp].status = '9';
	  break;
     case '1':
     case '0':   /* working.  send inop */
	  if (ship->torp[torp].status == TORP_ONLINE)
	       ship->num_torps_online--;
	  ship->torp[torp].status = TORP_DAMAGED;
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "%s %d damaged.  Inoperable.", capstr(ship->torp_string), 
		   torp);
	  ship->torp[torp].charge = 0;
	  ship->damage[SYSTEM_TORP_0 + torp].status = '7';
	  ship->damage[SYSTEM_TORP_0 + torp].time_to_fix = 240;
	  break;
     }

     return 0;
}

int damDamageEngine(TAG *object, int points)
{
     SHIP *ship = object->shipdata;
     dbref man;
     int space, output;
     XYZ pos;

     /* As this can destroy a ship - must check for being dead already */
     if (Removed(object))
	  return 1;

     ship->damage[SYSTEM_ENGINE].status += points;

     /* if the damage is greater than '8', zorch the ship */
     if (ship->damage[SYSTEM_ENGINE].status > '8') {
	  /* Engine destroyed, ship is killed */
	  NotifyExcept(dbrefEngine(ship), PSE_NOTHING, 
		       "Antimatter chamber breach.  Ship destroyed.");
	  NotifyExcept(dbrefBridge(ship), PSE_NOTHING, 
		       "Antimatter chamber breach.  Ship destroyed.");

	  evTrigger(object, EVENT_DESTROYED, "");

	  /* If in real space, log it */
	  if (space_info[object->space].flags & SPACE_LOGGED) {
	       log_space("%s(#%d) destroyed. Manned consoles:", object->name, 
			 object->data_object);

	       for (output = 0; output < ship->num_consoles; output ++) {
		    man = dbrefUser(ship->consoles[output].dbref);
		    log_space("Console %d(#%d): %s(#%d)",
			      output, ship->consoles[output].dbref,
			      getObjectName(man), man);
	       }
	  }

	  space = object->space;
	  pos.x = object->pos.x;
	  pos.y = object->pos.y;
	  pos.z = object->pos.z;
	  output = ship->reactor_output;

	  objRemoveObject(object);
	  damExplosionDamage(space, pos, output);

	  /* Return 1 to flag we just fragged the ship */
	  return 1;
     }

     /* The ship survived. */
     ship->damage[SYSTEM_ENGINE].time_to_fix = 600;
     ship->current_reactor_output_max = ship->reactor_output_max *
	  (1 - ((float)(ship->damage[SYSTEM_ENGINE].status - '0')
		* 0.1));

     MFNotify(ship, -1, CONS_ENG | CONS_DAM, CONS_ACTIVE,
	      "Engine damage.  New max power is %d.", 
	      ship->current_reactor_output_max);

     engAllocCheck(object);

     return 0;
}

int damDamageScanner(TAG *object, int points)
{
     SHIP *ship = object->shipdata;

     switch (ship->damage[SYSTEM_SCANNERS].status) {
     case '9':
	  break;
     case '7':
	  ship->damage[SYSTEM_SCANNERS].status = '9';
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Active scanners destroyed.");
	  break;

     case '6':
     case '1':
	  ship->damage[SYSTEM_SCANNERS].status = '7';
	  ship->damage[SYSTEM_SCANNERS].time_to_fix = 600;
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Active scanners damaged.  Offline.");
	  break;

     case '0':
	  ship->damage[SYSTEM_SCANNERS].status = '6';
	  ship->damage[SYSTEM_SCANNERS].time_to_fix = 300;
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Active scanners heavily damaged. Range reduced.");
	  ship->scanner_range /= 2;
	  break;
     }

     return 0;
}

int damDamageSensor(TAG *object, int points)
{
     SHIP *ship = object->shipdata;

     switch (ship->damage[SYSTEM_SENSORS].status) {
     case '9':
	  break;
     case '7':
	  ship->damage[SYSTEM_SENSORS].status = '9';
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Passive sensors destroyed.");
	  break;

     case '6':
     case '1':
	  ship->damage[SYSTEM_SENSORS].status = '7';
	  ship->damage[SYSTEM_SENSORS].time_to_fix = 600;
	  object->tagflags &= ~CAN_SENSE;
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Passive sensors damaged.  Inoperable.");
	  break;

     case '0':
	  ship->damage[SYSTEM_SENSORS].status = '7';
	  ship->damage[SYSTEM_SENSORS].time_to_fix = 300;
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Passive sensors damaged.  Range reduced.");
	  object->sensor_range *= 0.6;
	  break;
     }

     while (object->contact_list != NULL)
	  snsRemoveContact(object, object->contact_list->listref, 0);

     return 0;
}

int damDamageTransporter(TAG *object, int points)
{
     SHIP *ship = object->shipdata;

     switch (ship->damage[SYSTEM_TRANSPORTER].status) {
     case '9':
	  break;
     case '7':
	  ship->damage[SYSTEM_TRANSPORTER].status = '9';
	  MFNotify(ship, -1, CONS_TRANS | CONS_DAM, CONS_ACTIVE,
		   "Transporter destroyed.");
	  break;

     case '6':
     case '1':
	  ship->damage[SYSTEM_TRANSPORTER].status = '7';
	  ship->damage[SYSTEM_TRANSPORTER].time_to_fix = 600;
	  MFNotify(ship, -1, CONS_TRANS | CONS_DAM, CONS_ACTIVE,
		   "Transporter damaged.  Inoperable.");
	  break;

     case '0':
	  ship->damage[SYSTEM_TRANSPORTER].status = '6';
	  ship->damage[SYSTEM_TRANSPORTER].time_to_fix = 300;
	  MFNotify(ship, -1, CONS_TRANS | CONS_DAM, CONS_ACTIVE,
		   "Transporter damaged. Range reduced.");
	  object->transporter_range *= 0.5;
	  break;
     }

     return 0;
}

int damDamageTractor(TAG *object, int points)
{
     SHIP *ship = object->shipdata;

     switch (ship->damage[SYSTEM_TRACTOR].status) {
     case '9':
	  break;
     case '7':
	  ship->damage[SYSTEM_TRACTOR].status = '9';
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Tractor beam generator destroyed.");
	  break;

     case '6':
     case '1':
	  ship->damage[SYSTEM_TRACTOR].status = '7';
	  ship->damage[SYSTEM_TRACTOR].time_to_fix = 600;
	  if (ship->tractor_status == TRACTOR_ON ||
	      ship->tractor_status == TRACTOR_DRAW)
	       tacBreakTractor(object);
	  ship->tractor_status = TRACTOR_DAMAGED;
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Tractor beam generator damaged. Inoperable.");
	  break;

     case '0':
	  ship->damage[SYSTEM_TRACTOR].status = '6';
	  ship->damage[SYSTEM_TRACTOR].time_to_fix = 300;
	  MFNotify(ship, -1, CONS_TAC | CONS_DAM, CONS_ACTIVE,
		   "Tractor beam generator damaged. Effectiveness reduced.");
	  ship->tractor_effect *= 0.5;
	  ship->tractor_power *= 0.5;
	  break;
     }

     return 0;
}
